"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getParsedContent = getParsedContent;
exports.getConfig = getConfig;
exports.deleteNonDefaultConfig = deleteNonDefaultConfig;
const tslib_1 = require("tslib");
const url_1 = require("url");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const fs_extra_1 = tslib_1.__importDefault(require("fs-extra"));
const upath_1 = tslib_1.__importDefault(require("upath"));
const logger_1 = require("../../../../logger");
const common_1 = require("../../../../util/common");
const fs_1 = require("../../../../util/fs");
const yaml_1 = require("../../../../util/yaml");
const util_1 = require("./util");
async function getParsedContent(file) {
    if (upath_1.default.basename(file) === '.renovaterc') {
        return (0, common_1.parseJson)(await (0, fs_1.readSystemFile)(file, 'utf8'), file);
    }
    switch (upath_1.default.extname(file)) {
        case '.yaml':
        case '.yml':
            return (0, yaml_1.parseSingleYaml)(await (0, fs_1.readSystemFile)(file, 'utf8'));
        case '.json5':
        case '.json':
            return (0, common_1.parseJson)(await (0, fs_1.readSystemFile)(file, 'utf8'), file);
        case '.cjs':
        case '.mjs':
        case '.js': {
            const absoluteFilePath = upath_1.default.isAbsolute(file)
                ? file
                : `${process.cwd()}/${file}`;
            // use file url paths to avoid issues with windows paths
            // typescript does not support file URL for import
            const tmpConfig = await import((0, url_1.pathToFileURL)(absoluteFilePath).href);
            /* v8 ignore next -- not testable */
            let config = tmpConfig.default ?? tmpConfig;
            // Allow the config to be a function
            if (is_1.default.function(config)) {
                config = config();
            }
            return config;
        }
        default:
            throw new Error('Unsupported file type');
    }
}
async function getConfig(env) {
    const configFile = env.RENOVATE_CONFIG_FILE ?? 'config.js';
    const configFileExists = await fs_extra_1.default.pathExists(configFile);
    if (env.RENOVATE_CONFIG_FILE && !configFileExists) {
        logger_1.logger.fatal({ configFile }, `Custom config file specified in RENOVATE_CONFIG_FILE must exist`);
        process.exit(1);
    }
    let config = {};
    if (!configFileExists) {
        logger_1.logger.debug('No config file found on disk - skipping');
        return config;
    }
    logger_1.logger.debug('Checking for config file in ' + configFile);
    try {
        config = await getParsedContent(configFile);
    }
    catch (err) {
        if (err instanceof SyntaxError || err instanceof TypeError) {
            logger_1.logger.fatal({ error: err.stack }, 'Could not parse config file');
            process.exit(1);
        }
        else if (err instanceof ReferenceError) {
            logger_1.logger.fatal(`Error parsing config file due to unresolved variable(s): ${err.message}`);
            process.exit(1);
        }
        else if (err.message === 'Unsupported file type') {
            logger_1.logger.fatal(err.message);
            process.exit(1);
        }
        else if (env.RENOVATE_CONFIG_FILE) {
            logger_1.logger.debug({ err }, 'Parse error');
            logger_1.logger.fatal('Error parsing config file');
            process.exit(1);
        }
        logger_1.logger.debug('Error reading or parsing file - skipping');
    }
    if (is_1.default.nonEmptyObject(config.processEnv)) {
        const exportedKeys = [];
        for (const [key, value] of Object.entries(config.processEnv)) {
            if (!is_1.default.nonEmptyString(value)) {
                logger_1.logger.error({ key }, 'processEnv value is not a string.');
                continue;
            }
            exportedKeys.push(key);
            process.env[key] = value;
        }
        logger_1.logger.debug({ keys: exportedKeys }, 'processEnv keys were exported to env');
        delete config.processEnv;
    }
    return (0, util_1.migrateAndValidateConfig)(config, configFile);
}
async function deleteNonDefaultConfig(env, deleteConfigFile) {
    const configFile = env.RENOVATE_CONFIG_FILE;
    if (is_1.default.undefined(configFile) || is_1.default.emptyStringOrWhitespace(configFile)) {
        return;
    }
    if (!deleteConfigFile) {
        return;
    }
    if (!(await fs_extra_1.default.pathExists(configFile))) {
        return;
    }
    try {
        await fs_extra_1.default.remove(configFile);
        logger_1.logger.trace({ path: configFile }, 'config file successfully deleted');
    }
    catch (err) {
        logger_1.logger.warn({ err }, 'error deleting config file');
    }
}
//# sourceMappingURL=file.js.map