"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.joinUrlParts = joinUrlParts;
exports.ensurePathPrefix = ensurePathPrefix;
exports.ensureTrailingSlash = ensureTrailingSlash;
exports.trimTrailingSlash = trimTrailingSlash;
exports.trimLeadingSlash = trimLeadingSlash;
exports.trimSlashes = trimSlashes;
exports.resolveBaseUrl = resolveBaseUrl;
exports.replaceUrlPath = replaceUrlPath;
exports.getQueryString = getQueryString;
exports.isHttpUrl = isHttpUrl;
exports.parseUrl = parseUrl;
exports.createURLFromHostOrURL = createURLFromHostOrURL;
exports.parseLinkHeader = parseLinkHeader;
exports.massageHostUrl = massageHostUrl;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
// eslint-disable-next-line no-restricted-imports
const parse_link_header_1 = tslib_1.__importDefault(require("parse-link-header"));
const url_join_1 = tslib_1.__importDefault(require("url-join"));
const logger_1 = require("../logger");
const regex_1 = require("./regex");
function joinUrlParts(...parts) {
    return (0, url_join_1.default)(...parts);
}
function ensurePathPrefix(url, prefix) {
    const parsed = new URL(url);
    const fullPath = parsed.pathname + parsed.search;
    if (fullPath.startsWith(prefix)) {
        return url;
    }
    return parsed.origin + prefix + fullPath;
}
function ensureTrailingSlash(url) {
    return url.replace(/\/?$/, '/'); // TODO #12875 adds slash at the front when re2 is used
}
function trimTrailingSlash(url) {
    return url.replace((0, regex_1.regEx)(/\/+$/), '');
}
function trimLeadingSlash(path) {
    return path.replace(/^\/+/, '');
}
function trimSlashes(path) {
    return trimLeadingSlash(trimTrailingSlash(path));
}
/**
 * Resolves an input path against a base URL
 *
 * @param baseUrl - base URL to resolve against
 * @param input - input path (if this is a full URL, it will be returned)
 */
function resolveBaseUrl(baseUrl, input) {
    const inputString = input.toString();
    let host;
    let pathname;
    try {
        ({ host, pathname } = new URL(inputString));
    }
    catch {
        pathname = inputString;
    }
    return host ? inputString : (0, url_join_1.default)(baseUrl, pathname || '');
}
/**
 * Replaces the path of a URL with a new path
 *
 * @param baseUrl - source URL
 * @param path - replacement path (if this is a full URL, it will be returned)
 */
function replaceUrlPath(baseUrl, path) {
    if (parseUrl(path)) {
        return path;
    }
    const { origin } = is_1.default.string(baseUrl) ? new URL(baseUrl) : baseUrl;
    return (0, url_join_1.default)(origin, path);
}
function getQueryString(params) {
    const usp = new URLSearchParams();
    for (const [k, v] of Object.entries(params)) {
        if (is_1.default.array(v)) {
            for (const item of v) {
                // TODO: fix me?
                // eslint-disable-next-line @typescript-eslint/no-base-to-string
                usp.append(k, item.toString());
            }
        }
        else {
            usp.append(k, v.toString());
        }
    }
    return usp.toString();
}
function isHttpUrl(url) {
    if (!is_1.default.nonEmptyString(url) && !is_1.default.urlInstance(url)) {
        return false;
    }
    const protocol = parseUrl(url)?.protocol;
    return protocol === 'https:' || protocol === 'http:';
}
function parseUrl(url) {
    if (!url) {
        return null;
    }
    if (url instanceof URL) {
        return url;
    }
    try {
        return new URL(url);
    }
    catch {
        return null;
    }
}
/**
 * Tries to create an URL object from either a full URL string or a hostname
 * @param url either the full url or a hostname
 * @returns an URL object or null
 */
function createURLFromHostOrURL(url) {
    return parseUrl(url) ?? parseUrl(`https://${url}`);
}
function parseLinkHeader(linkHeader) {
    if (!is_1.default.nonEmptyString(linkHeader)) {
        return null;
    }
    if (linkHeader.length > 2000) {
        logger_1.logger.warn({ linkHeader }, 'Link header too long.');
        return null;
    }
    return (0, parse_link_header_1.default)(linkHeader);
}
/**
 * prefix https:// to hosts with port or path
 */
function massageHostUrl(url) {
    if (!url.includes('://') && url.includes('/')) {
        return 'https://' + url;
    }
    else if (!url.includes('://') && url.includes(':')) {
        return 'https://' + url;
    }
    else {
        return url;
    }
}
//# sourceMappingURL=url.js.map