"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UUIDRegex = void 0;
exports.isDockerDigest = isDockerDigest;
exports.getRegexOrGlobPredicate = getRegexOrGlobPredicate;
exports.matchRegexOrGlob = matchRegexOrGlob;
exports.matchRegexOrGlobList = matchRegexOrGlobList;
exports.anyMatchRegexOrGlobList = anyMatchRegexOrGlobList;
exports.isRegexMatch = isRegexMatch;
exports.getRegexPredicate = getRegexPredicate;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const minimatch_1 = require("./minimatch");
const regex_1 = require("./regex");
function isDockerDigest(input) {
    return /^sha256:[a-f0-9]{64}$/i.test(input);
}
function getRegexOrGlobPredicate(pattern) {
    const regExPredicate = getRegexPredicate(pattern);
    if (regExPredicate) {
        return regExPredicate;
    }
    const mm = (0, minimatch_1.minimatch)(pattern, { dot: true, nocase: true });
    return (x) => mm.match(x);
}
function matchRegexOrGlob(input, pattern) {
    if (pattern === '*') {
        return true;
    }
    const predicate = getRegexOrGlobPredicate(pattern);
    return predicate(input);
}
function matchRegexOrGlobList(input, patterns) {
    if (!patterns.length) {
        return false;
    }
    // Return false if there are positive patterns and none match
    const positivePatterns = patterns.filter((pattern) => !pattern.startsWith('!'));
    if (positivePatterns.length &&
        !positivePatterns.some((pattern) => matchRegexOrGlob(input, pattern))) {
        return false;
    }
    // Every negative pattern must be true to return true
    const negativePatterns = patterns.filter((pattern) => pattern.startsWith('!'));
    if (negativePatterns.length &&
        !negativePatterns.every((pattern) => matchRegexOrGlob(input, pattern))) {
        return false;
    }
    return true;
}
function anyMatchRegexOrGlobList(inputs, patterns) {
    return inputs.some((input) => matchRegexOrGlobList(input, patterns));
}
exports.UUIDRegex = (0, regex_1.regEx)(/^[0-9a-f]{8}-[0-9a-f]{4}-[1-5][0-9a-f]{3}-[89ab][0-9a-f]{3}-[0-9a-f]{12}$/i);
const configValStart = (0, regex_1.regEx)(/^!?\//);
const configValEnd = (0, regex_1.regEx)(/\/i?$/);
function isRegexMatch(input) {
    return (is_1.default.string(input) && configValStart.test(input) && configValEnd.test(input));
}
function parseRegexMatch(input) {
    try {
        const regexString = input
            .replace(configValStart, '')
            .replace(configValEnd, '');
        return input.endsWith('i') ? (0, regex_1.regEx)(regexString, 'i') : (0, regex_1.regEx)(regexString);
    }
    catch {
        // no-op
    }
    return null;
}
function getRegexPredicate(input) {
    if (isRegexMatch(input)) {
        const configRegex = parseRegexMatch(input);
        if (configRegex) {
            const isPositive = !input.startsWith('!');
            return (x) => {
                const res = configRegex.test(x);
                return isPositive ? res : !res;
            };
        }
    }
    return null;
}
//# sourceMappingURL=string-match.js.map