import { DateTime } from 'luxon';
import type { JsonArray, JsonValue } from 'type-fest';
import { type ZodEffects, type ZodString, type ZodType, type ZodTypeDef, z } from 'zod';
import type { PackageDependency } from '../../modules/manager/types';
import type { YamlOptions } from '../yaml';
interface ErrorContext<T> {
    error: z.ZodError;
    input: T;
}
interface LooseOpts<T> {
    onError?: (err: ErrorContext<T>) => void;
}
/**
 * Works like `z.array()`, but drops wrong elements instead of invalidating the whole array.
 *
 * **Important**: non-array inputs are still invalid.
 * Use `LooseArray(...).catch([])` to handle it.
 *
 * @param Elem Schema for array elements
 * @param onError Callback for errors
 * @returns Schema for array
 */
export declare function LooseArray<Schema extends z.ZodTypeAny>(Elem: Schema, { onError }?: LooseOpts<unknown[]>): z.ZodEffects<z.ZodArray<z.ZodAny, 'many'>, z.TypeOf<Schema>[], any[]>;
type LooseRecordResult<KeySchema extends z.ZodTypeAny, ValueSchema extends z.ZodTypeAny> = z.ZodEffects<z.ZodRecord<z.ZodString, z.ZodAny>, Record<z.TypeOf<KeySchema>, z.TypeOf<ValueSchema>>, Record<z.TypeOf<KeySchema>, any>>;
type LooseRecordOpts<KeySchema extends z.ZodTypeAny, ValueSchema extends z.ZodTypeAny> = LooseOpts<Record<z.TypeOf<KeySchema> | z.TypeOf<ValueSchema>, unknown>>;
/**
 * Works like `z.record()`, but drops wrong elements instead of invalidating the whole record.
 *
 * **Important**: non-record inputs other are still invalid.
 * Use `LooseRecord(...).catch({})` to handle it.
 *
 * @param KeyValue Schema for record keys
 * @param ValueValue Schema for record values
 * @param onError Callback for errors
 * @returns Schema for record
 */
export declare function LooseRecord<ValueSchema extends z.ZodTypeAny>(Value: ValueSchema): LooseRecordResult<z.ZodString, ValueSchema>;
export declare function LooseRecord<KeySchema extends z.ZodTypeAny, ValueSchema extends z.ZodTypeAny>(Key: KeySchema, Value: ValueSchema): LooseRecordResult<KeySchema, ValueSchema>;
export declare function LooseRecord<ValueSchema extends z.ZodTypeAny>(Value: ValueSchema, { onError }: LooseRecordOpts<z.ZodString, ValueSchema>): LooseRecordResult<z.ZodString, ValueSchema>;
export declare function LooseRecord<KeySchema extends z.ZodTypeAny, ValueSchema extends z.ZodTypeAny>(Key: KeySchema, Value: ValueSchema, { onError }: LooseRecordOpts<KeySchema, ValueSchema>): LooseRecordResult<KeySchema, ValueSchema>;
export declare const Json: ZodEffects<ZodString, string | number | boolean | import("type-fest").JsonObject | JsonValue[] | readonly JsonValue[] | null, string>;
export declare const Json5: ZodEffects<ZodString, string | number | boolean | import("type-fest").JsonObject | JsonValue[] | readonly JsonValue[] | null, string>;
export declare const Jsonc: ZodEffects<ZodString, string | number | boolean | import("type-fest").JsonObject | JsonValue[] | readonly JsonValue[] | null, string>;
export declare const UtcDate: ZodEffects<ZodString, DateTime<boolean>, string>;
export declare const Yaml: ZodEffects<ZodString, string | number | boolean | import("type-fest").JsonObject | JsonValue[] | readonly JsonValue[] | null, string>;
export declare const MultidocYaml: ZodEffects<ZodString, JsonValue[] | readonly JsonValue[], string>;
export declare function multidocYaml(opts?: Omit<YamlOptions, 'customSchema'>): ZodEffects<ZodString, JsonArray, string>;
export declare const Toml: ZodEffects<ZodString, unknown, string>;
export declare function withDepType<Output extends PackageDependency[], Schema extends ZodType<Output, ZodTypeDef, unknown>>(schema: Schema, depType: string, force?: boolean): ZodEffects<Schema>;
export declare function withDebugMessage<Input, Output>(value: Output, msg: string): (ctx: {
    error: z.ZodError;
    input: Input;
}) => Output;
export declare function withTraceMessage<Input, Output>(value: Output, msg: string): (ctx: {
    error: z.ZodError;
    input: Input;
}) => Output;
export declare const NotCircular: ZodEffects<z.ZodUnknown, unknown, unknown>;
export {};
