"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.toMs = toMs;
exports.satisfiesDateRange = satisfiesDateRange;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const luxon_1 = require("luxon");
const ms_1 = tslib_1.__importDefault(require("ms"));
const logger_1 = require("../logger");
const regex_1 = require("./regex");
const splitRegex = (0, regex_1.regEx)(/(.*?[a-z]+)/);
function split(time) {
    return time
        .split(splitRegex)
        .map((x) => x.trim())
        .filter(is_1.default.nonEmptyString);
}
function applyCustomFormat(spec) {
    const monthRegex = (0, regex_1.regEx)(/^(\d+)\s*(?:months?|M)$/);
    return spec.replace(monthRegex, (_, months) => `${months * 30} days`);
}
function toMs(time) {
    try {
        const specs = split(time).map(applyCustomFormat);
        if (!specs.length) {
            logger_1.logger.debug({ time }, `Invalid time specifier: '${time}'`);
            return null;
        }
        let totalMillis = 0;
        for (const spec of specs) {
            const millis = (0, ms_1.default)(spec);
            if (!is_1.default.number(millis)) {
                logger_1.logger.debug({ time }, `Invalid time specifier: '${spec}'`);
                return null;
            }
            totalMillis += millis;
        }
        return totalMillis;
    }
    catch (err) {
        logger_1.logger.debug({ time, err }, `Invalid time specifier: '${time}'`);
        return null;
    }
}
const rangeRegex = (0, regex_1.regEx)(/^(?<operator>(>=|<=|<|>))\s*(?<age>.*)$/);
function satisfiesDateRange(date, range) {
    const grps = range.trim().match(rangeRegex)?.groups;
    if (!grps) {
        return null;
    }
    const { operator, age } = grps;
    const luxonDate = luxon_1.DateTime.fromISO(date, { zone: 'utc' });
    if (!luxonDate.isValid) {
        logger_1.logger.trace(`Invalid date when computing satisfiesDateRange: '${date}'`);
        return null;
    }
    const dateMs = luxonDate.toMillis();
    const ageMs = toMs(age);
    if (!is_1.default.number(ageMs)) {
        return null;
    }
    const rangeMs = Date.now() - ageMs;
    switch (operator) {
        case '>':
            return dateMs < rangeMs;
        case '>=':
            return dateMs <= rangeMs;
        case '<':
            return dateMs > rangeMs;
        case '<=':
            return dateMs >= rangeMs;
        // istanbul ignore next: can never happen
        default:
            return dateMs === rangeMs;
    }
}
//# sourceMappingURL=pretty-time.js.map