"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.applyPackageRules = applyPackageRules;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const slugify_1 = tslib_1.__importDefault(require("slugify"));
const config_1 = require("../../config");
const logger_1 = require("../../logger");
const template_1 = require("../template");
const matchers_1 = tslib_1.__importDefault(require("./matchers"));
async function matchesRule(inputConfig, packageRule) {
    for (const matcher of matchers_1.default) {
        const isMatch = await matcher.matches(inputConfig, packageRule);
        // no rules are defined
        if (is_1.default.nullOrUndefined(isMatch)) {
            continue;
        }
        if (!is_1.default.truthy(isMatch)) {
            return false;
        }
    }
    return true;
}
async function applyPackageRules(inputConfig, stageName) {
    let config = { ...inputConfig };
    const packageRules = config.packageRules ?? [];
    logger_1.logger.trace({ dependency: config.depName, packageRules }, `Checking against ${packageRules.length} packageRules`);
    for (const packageRule of packageRules) {
        // This rule is considered matched if there was at least one positive match and no negative matches
        if (await matchesRule(config, packageRule)) {
            // Package rule config overrides any existing config
            const toApply = removeMatchers({ ...packageRule });
            if (config.groupSlug && packageRule.groupName && !packageRule.groupSlug) {
                // Need to apply groupSlug otherwise the existing one will take precedence
                toApply.groupSlug = (0, slugify_1.default)(packageRule.groupName, {
                    lower: true,
                });
            }
            if (toApply.enabled === false && config.enabled !== false) {
                config.skipReason = 'package-rules';
                if (stageName) {
                    config.skipStage = stageName;
                }
            }
            if (toApply.enabled === true && config.enabled === false) {
                delete config.skipReason;
                delete config.skipStage;
            }
            if (is_1.default.string(toApply.overrideDatasource) &&
                toApply.overrideDatasource !== config.datasource) {
                logger_1.logger.debug(`Overriding datasource from ${config.datasource} to ${toApply.overrideDatasource} for ${config.depName}`);
                config.datasource = toApply.overrideDatasource;
            }
            if (is_1.default.string(toApply.overrideDepName) &&
                toApply.overrideDepName !== config.depName) {
                logger_1.logger.debug(`Overriding depName from ${config.depName} to ${toApply.overrideDepName}`);
                config.depName = (0, template_1.compile)(toApply.overrideDepName, config);
            }
            if (is_1.default.string(toApply.overridePackageName) &&
                toApply.overridePackageName !== config.packageName) {
                logger_1.logger.debug(`Overriding packageName from ${config.packageName} to ${toApply.overridePackageName} for ${config.depName}`);
                config.packageName = (0, template_1.compile)(toApply.overridePackageName, config);
            }
            delete toApply.overrideDatasource;
            delete toApply.overrideDepName;
            delete toApply.overridePackageName;
            config = (0, config_1.mergeChildConfig)(config, toApply);
        }
    }
    return config;
}
function removeMatchers(packageRule) {
    for (const key of Object.keys(packageRule)) {
        if (key.startsWith('match') || key.startsWith('exclude')) {
            delete packageRule[key];
        }
    }
    return packageRule;
}
//# sourceMappingURL=index.js.map