"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.setHttpRateLimits = setHttpRateLimits;
exports.getConcurrentRequestsLimit = getConcurrentRequestsLimit;
exports.getThrottleIntervalMs = getThrottleIntervalMs;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const host_rules_1 = require("../host-rules");
const hostRules = tslib_1.__importStar(require("../host-rules"));
// The first match wins
const concurrencyDefaults = [
    {
        matchHost: 'registry.npmjs.org',
        concurrency: 999,
    },
    {
        matchHost: 'repology.org',
        concurrency: 1,
    },
    {
        matchHost: '*',
        concurrency: 16,
    },
];
// The first match wins
const throttleDefaults = [
    {
        // https://guides.rubygems.org/rubygems-org-rate-limits/
        matchHost: 'rubygems.org',
        throttleMs: 125,
    },
    {
        // https://crates.io/data-access#api
        matchHost: 'https://crates.io/api/',
        throttleMs: 1000,
    },
    {
        // The rate limit is 100 per second, according this comment:
        // https://github.com/renovatebot/renovate/discussions/27018#discussioncomment-10336270
        //
        // We stick to 20 per second just in case.
        matchHost: 'https://plugins.gradle.org',
        throttleMs: 50,
    },
    {
        matchHost: 'repology.org',
        throttleMs: 2000,
    },
];
let throttleLimits = [];
let concurrencyLimits = [];
function setHttpRateLimits(concurrencyRules, throttleRules) {
    concurrencyLimits = concurrencyRules ?? concurrencyDefaults;
    throttleLimits = throttleRules ?? throttleDefaults;
}
function matches(url, host) {
    if (host === '*') {
        return true;
    }
    return (0, host_rules_1.matchesHost)(url, host);
}
function getConcurrentRequestsLimit(url) {
    let result = null;
    const { concurrentRequestLimit: hostRuleLimit } = hostRules.find({ url });
    if (is_1.default.number(hostRuleLimit) &&
        hostRuleLimit > 0 &&
        hostRuleLimit < Number.MAX_SAFE_INTEGER) {
        result = hostRuleLimit;
    }
    for (const { matchHost, concurrency: limit } of concurrencyLimits) {
        if (!matches(url, matchHost)) {
            continue;
        }
        if (result && result <= limit) {
            continue;
        }
        result = limit;
        break;
    }
    return result;
}
function getThrottleIntervalMs(url) {
    let result = null;
    const { maxRequestsPerSecond } = hostRules.find({ url });
    if (is_1.default.number(maxRequestsPerSecond) && maxRequestsPerSecond > 0) {
        result = Math.ceil(1000 / maxRequestsPerSecond);
    }
    for (const { matchHost, throttleMs: limit } of throttleLimits) {
        if (!matches(url, matchHost)) {
            continue;
        }
        if (result && result >= limit) {
            continue;
        }
        result = limit;
        break;
    }
    return result;
}
//# sourceMappingURL=rate-limits.js.map