"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PackageHttpCacheProvider = void 0;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const luxon_1 = require("luxon");
const global_1 = require("../../../config/global");
const package_1 = require("../../cache/package");
const ttl_1 = require("../../cache/package/ttl");
const regex_1 = require("../../regex");
const stats_1 = require("../../stats");
const abstract_http_cache_provider_1 = require("./abstract-http-cache-provider");
class PackageHttpCacheProvider extends abstract_http_cache_provider_1.AbstractHttpCacheProvider {
    namespace;
    softTtlMinutes;
    hardTtlMinutes;
    checkCacheControlHeader;
    checkAuthorizationHeader;
    constructor({ namespace, softTtlMinutes = 15, checkCacheControlHeader = false, checkAuthorizationHeader = false, }) {
        super();
        this.namespace = namespace;
        const ttl = (0, ttl_1.resolveTtlValues)(this.namespace, softTtlMinutes);
        this.softTtlMinutes = ttl.softTtlMinutes;
        this.hardTtlMinutes = ttl.hardTtlMinutes;
        this.checkCacheControlHeader = checkCacheControlHeader;
        this.checkAuthorizationHeader = checkAuthorizationHeader;
    }
    async load(url) {
        return await (0, package_1.get)(this.namespace, url);
    }
    async persist(url, data) {
        await (0, package_1.set)(this.namespace, url, data, this.hardTtlMinutes);
    }
    async bypassServer(url, ignoreSoftTtl = false) {
        const cached = await this.get(url);
        if (!cached) {
            return null;
        }
        if (ignoreSoftTtl) {
            return cached.httpResponse;
        }
        const cachedAt = luxon_1.DateTime.fromISO(cached.timestamp);
        const deadline = cachedAt.plus({ minutes: this.softTtlMinutes });
        const now = luxon_1.DateTime.now();
        if (now >= deadline) {
            stats_1.HttpCacheStats.incLocalMisses(url);
            return null;
        }
        stats_1.HttpCacheStats.incLocalHits(url);
        return cached.httpResponse;
    }
    cacheAllowed(resp) {
        const allowedViaGlobalConfig = global_1.GlobalConfig.get('cachePrivatePackages', false);
        if (allowedViaGlobalConfig) {
            return true;
        }
        if (this.checkCacheControlHeader &&
            is_1.default.string(resp.headers['cache-control'])) {
            const isPublic = resp.headers['cache-control']
                .toLocaleLowerCase()
                .split((0, regex_1.regEx)(/\s*,\s*/))
                .includes('public');
            if (!isPublic) {
                return false;
            }
        }
        if (this.checkAuthorizationHeader && resp.authorization) {
            return false;
        }
        return true;
    }
    async wrapServerResponse(url, resp) {
        if (resp.statusCode === 200 && !this.cacheAllowed(resp)) {
            return resp;
        }
        return await super.wrapServerResponse(url, resp);
    }
}
exports.PackageHttpCacheProvider = PackageHttpCacheProvider;
//# sourceMappingURL=package-http-cache-provider.js.map