"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AbstractHttpCacheProvider = void 0;
const logger_1 = require("../../../logger");
const stats_1 = require("../../stats");
const util_1 = require("../util");
const schema_1 = require("./schema");
class AbstractHttpCacheProvider {
    async get(url) {
        const cache = await this.load(url);
        const httpCache = schema_1.HttpCacheSchema.parse(cache);
        if (!httpCache) {
            return null;
        }
        return httpCache;
    }
    async setCacheHeaders(url, opts) {
        const httpCache = await this.get(url);
        if (!httpCache) {
            return;
        }
        opts.headers ??= {};
        if (httpCache.etag) {
            opts.headers['If-None-Match'] = httpCache.etag;
        }
        if (httpCache.lastModified) {
            opts.headers['If-Modified-Since'] = httpCache.lastModified;
        }
    }
    bypassServer(_url, _ignoreSoftTtl) {
        return Promise.resolve(null);
    }
    async wrapServerResponse(url, resp) {
        if (resp.statusCode === 200) {
            const etag = resp.headers?.etag;
            const lastModified = resp.headers?.['last-modified'];
            stats_1.HttpCacheStats.incRemoteMisses(url);
            const httpResponse = (0, util_1.copyResponse)(resp, true);
            const timestamp = new Date().toISOString();
            const newHttpCache = schema_1.HttpCacheSchema.parse({
                etag,
                lastModified,
                httpResponse,
                timestamp,
            });
            /* v8 ignore start: should never happen */
            if (!newHttpCache) {
                logger_1.logger.debug(`http cache: failed to persist cache for ${url}`);
                return resp;
            } /* v8 ignore stop */
            logger_1.logger.debug(`http cache: saving ${url} (etag=${etag}, lastModified=${lastModified})`);
            await this.persist(url, newHttpCache);
            return resp;
        }
        if (resp.statusCode === 304) {
            const httpCache = await this.get(url);
            if (!httpCache) {
                return resp;
            }
            const timestamp = httpCache.timestamp;
            logger_1.logger.debug(`http cache: Using cached response: ${url} from ${timestamp}`);
            stats_1.HttpCacheStats.incRemoteHits(url);
            const cachedResp = (0, util_1.copyResponse)(httpCache.httpResponse, true);
            cachedResp.authorization = resp.authorization;
            return cachedResp;
        }
        return resp;
    }
}
exports.AbstractHttpCacheProvider = AbstractHttpCacheProvider;
//# sourceMappingURL=abstract-http-cache-provider.js.map