"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.adapter = void 0;
const zod_1 = require("zod");
const timestamp_1 = require("../../../timestamp");
const util_1 = require("../util");
const key = 'github-tags-datasource-v2';
const GithubGraphqlTag = zod_1.z.object({
    version: zod_1.z.string(),
    target: zod_1.z.union([
        zod_1.z.object({
            type: zod_1.z.literal('Commit'),
            oid: zod_1.z.string(),
            releaseTimestamp: timestamp_1.Timestamp,
        }),
        zod_1.z.object({
            type: zod_1.z.literal('Tag'),
            target: zod_1.z.union([
                zod_1.z.object({
                    type: zod_1.z.literal('Commit'),
                    oid: zod_1.z.string(),
                }),
                zod_1.z.object({
                    type: zod_1.z.literal('Tag'),
                    target: zod_1.z.object({ oid: zod_1.z.string() }),
                }),
            ]),
            tagger: zod_1.z.object({
                releaseTimestamp: timestamp_1.Timestamp,
            }),
        }),
    ]),
});
const query = (0, util_1.prepareQuery)(`
  refs(
    first: $count
    after: $cursor
    orderBy: {field: TAG_COMMIT_DATE, direction: DESC}
    refPrefix: "refs/tags/"
  ) {
    pageInfo {
      hasNextPage
      endCursor
    }
    nodes {
      version: name
      target {
        type: __typename
        ... on Commit {
          oid
          releaseTimestamp: committedDate
        }
        ... on Tag {
          target {
            type: __typename
            ... on Commit {
              oid
            }
            ... on Tag {
              target {
                oid
              }
            }
          }
          tagger {
            releaseTimestamp: date
          }
        }
      }
    }
  }`);
function transform(item) {
    const res = GithubGraphqlTag.safeParse(item);
    if (!res.success) {
        return null;
    }
    const { version, target } = item;
    if (target.type === 'Commit') {
        const releaseTimestamp = target.releaseTimestamp;
        const hash = target.oid;
        return { version, gitRef: version, hash, releaseTimestamp };
    }
    const releaseTimestamp = target.tagger.releaseTimestamp;
    if (target.target.type === 'Commit') {
        const hash = target.target.oid;
        return { version, gitRef: version, hash, releaseTimestamp };
    }
    const hash = target.target.target.oid;
    return { version, gitRef: version, hash, releaseTimestamp };
}
exports.adapter = { key, query, transform };
//# sourceMappingURL=tags-query-adapter.js.map