"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.adapter = void 0;
const zod_1 = require("zod");
const timestamp_1 = require("../../../timestamp");
const util_1 = require("../util");
const key = 'github-releases-datasource-v2';
const query = (0, util_1.prepareQuery)(`
  releases(
    first: $count
    after: $cursor
    orderBy: {field: CREATED_AT, direction: DESC}
  ) {
    pageInfo {
      hasNextPage
      endCursor
    }
    nodes {
      version: tagName
      releaseTimestamp: publishedAt
      isDraft
      isPrerelease
      url
      id: databaseId
      name
      description
    }
  }
`);
const GithubGraphqlRelease = zod_1.z.object({
    version: zod_1.z.string(),
    releaseTimestamp: timestamp_1.Timestamp,
    isDraft: zod_1.z.boolean(),
    isPrerelease: zod_1.z.boolean(),
    url: zod_1.z.string(),
    id: zod_1.z.number().nullable(),
    name: zod_1.z.string().nullable(),
    description: zod_1.z.string().nullable(),
});
function transform(item) {
    const releaseItem = GithubGraphqlRelease.safeParse(item);
    if (!releaseItem.success) {
        return null;
    }
    const { version, releaseTimestamp, isDraft, isPrerelease, url, id, name, description, } = releaseItem.data;
    if (isDraft) {
        return null;
    }
    const result = {
        version,
        releaseTimestamp,
        url,
    };
    if (id) {
        result.id = id;
    }
    if (name) {
        result.name = name;
    }
    if (description) {
        result.description = description;
    }
    if (isPrerelease) {
        result.isStable = false;
    }
    return result;
}
exports.adapter = { key, query, transform };
//# sourceMappingURL=releases-query-adapter.js.map