import type { RenovateConfig } from '../../config/types';
import type { GitProtocol } from '../../types/git';
import type { CommitFilesConfig, CommitResult, LongCommitSha, PushFilesConfig, StatusResult, StorageConfig, TreeItem } from './types';
export { setNoVerify } from './config';
export { setPrivateKey } from './private-key';
export declare const RENOVATE_FORK_UPSTREAM = "renovate-fork-upstream";
export declare function gitRetry<T>(gitFunc: () => Promise<T>): Promise<T>;
export declare const GIT_MINIMUM_VERSION = "2.33.0";
export declare function validateGitVersion(): Promise<boolean>;
export declare function fetchRevSpec(revSpec: string): Promise<void>;
export declare function initRepo(args: StorageConfig): Promise<void>;
export declare function resetToCommit(commit: LongCommitSha): Promise<void>;
export declare function setGitAuthor(gitAuthor: string | undefined): void;
export declare function writeGitAuthor(): Promise<void>;
export declare function setUserRepoConfig({ gitIgnoredAuthors, gitAuthor, }: RenovateConfig): void;
export declare function getSubmodules(): Promise<string[]>;
export declare function cloneSubmodules(shouldClone: boolean, cloneSubmodulesFilter: string[] | undefined): Promise<void>;
export declare function isCloned(): boolean;
export declare function syncGit(): Promise<void>;
export declare function getRepoStatus(path?: string): Promise<StatusResult>;
export declare function branchExists(branchName: string): boolean;
export declare function getBranchCommit(branchName: string): LongCommitSha | null;
export declare function getCommitMessages(): Promise<string[]>;
export declare function checkoutBranch(branchName: string): Promise<LongCommitSha>;
export declare function checkoutBranchFromRemote(branchName: string, remoteName: string): Promise<LongCommitSha>;
export declare function resetHardFromRemote(remoteAndBranch: string): Promise<void>;
export declare function forcePushToRemote(branchName: string, remote: string): Promise<void>;
export declare function getFileList(): Promise<string[]>;
export declare function getBranchList(): string[];
export declare function isBranchBehindBase(branchName: string, baseBranch: string): Promise<boolean>;
export declare function isBranchModified(branchName: string, baseBranch: string): Promise<boolean>;
export declare function isBranchConflicted(baseBranch: string, branch: string): Promise<boolean>;
export declare function deleteBranch(branchName: string): Promise<void>;
export declare function mergeToLocal(refSpecToMerge: string): Promise<void>;
export declare function mergeBranch(branchName: string): Promise<void>;
export declare function getBranchLastCommitTime(branchName: string): Promise<Date>;
export declare function getBranchFiles(branchName: string): Promise<string[] | null>;
export declare function getBranchFilesFromCommit(referenceCommit: LongCommitSha): Promise<string[] | null>;
export declare function getFile(filePath: string, branchName?: string): Promise<string | null>;
export declare function getFiles(fileNames: string[]): Promise<Record<string, string | null>>;
export declare function hasDiff(sourceRef: string, targetRef: string): Promise<boolean>;
/**
 *
 * Prepare local branch with commit
 *
 * 0. Hard reset
 * 1. Creates local branch with `origin/` prefix
 * 2. Perform `git add` (respecting mode) and `git remove` for each file
 * 3. Perform commit
 * 4. Check whether resulting commit is empty or not (due to .gitignore)
 * 5. If not empty, return commit info for further processing
 *
 */
export declare function prepareCommit({ branchName, files, message, force, }: CommitFilesConfig): Promise<CommitResult | null>;
export declare function pushCommit({ sourceRef, targetRef, files, pushOptions, }: PushFilesConfig): Promise<boolean>;
export declare function fetchBranch(branchName: string): Promise<LongCommitSha | null>;
export declare function commitFiles(commitConfig: CommitFilesConfig): Promise<LongCommitSha | null>;
export declare function getUrl({ protocol, auth, hostname, host, repository, }: {
    protocol?: GitProtocol;
    auth?: string;
    hostname?: string;
    host?: string;
    repository: string;
}): string;
/**
 *
 * Non-branch refs allow us to store git objects without triggering CI pipelines.
 * It's useful for API-based branch rebasing.
 *
 * @see https://stackoverflow.com/questions/63866947/pushing-git-non-branch-references-to-a-remote/63868286
 *
 */
export declare function pushCommitToRenovateRef(commitSha: string, refName: string): Promise<void>;
/**
 *
 * Removes all remote "refs/renovate/branches/*" refs in two steps:
 *
 * Step 1: list refs
 *
 *   $ git ls-remote origin "refs/renovate/branches/*"
 *
 *   > cca38e9ea6d10946bdb2d0ca5a52c205783897aa        refs/renovate/branches/foo
 *   > 29ac154936c880068994e17eb7f12da7fdca70e5        refs/renovate/branches/bar
 *   > 3fafaddc339894b6d4f97595940fd91af71d0355        refs/renovate/branches/baz
 *   > ...
 *
 * Step 2:
 *
 *   $ git push --delete origin refs/renovate/branches/foo refs/renovate/branches/bar refs/renovate/branches/baz
 *
 * If Step 2 fails because the repo doesn't allow bulk changes, we'll remove them one by one instead:
 *
 *   $ git push --delete origin refs/renovate/branches/foo
 *   $ git push --delete origin refs/renovate/branches/bar
 *   $ git push --delete origin refs/renovate/branches/baz
 */
export declare function clearRenovateRefs(): Promise<void>;
/**
 *
 * Obtain top-level items of commit tree.
 * We don't need subtree items, so here are 2 steps only.
 *
 * Step 1: commit SHA -> tree SHA
 *
 *   $ git cat-file -p <commit-sha>
 *
 *   > tree <tree-sha>
 *   > parent 59b8b0e79319b7dc38f7a29d618628f3b44c2fd7
 *   > ...
 *
 * Step 2: tree SHA -> tree items (top-level)
 *
 *   $ git cat-file -p <tree-sha>
 *
 *   > 040000 tree 389400684d1f004960addc752be13097fe85d776    src
 *   > ...
 *   > 100644 blob 7d2edde437ad4e7bceb70dbfe70e93350d99c98b    package.json
 *
 */
export declare function listCommitTree(commitSha: LongCommitSha): Promise<TreeItem[]>;
/**
 * Synchronize a forked branch with its upstream counterpart.
 *
 * syncForkWithUpstream updates the fork's branch, to match the corresponding branch in the upstream repository.
 * The steps are:
 * 1. Check if the branch exists locally.
 * 2. If the branch exists locally: checkout the local branch.
 * 3. If the branch does _not_ exist locally: checkout the upstream branch.
 * 4. Reset the local branch to match the upstream branch.
 * 5. Force push the (updated) local branch to the origin repository.
 *
 * @param {string} branchName - The name of the branch to synchronize.
 * @returns A promise that resolves to True if the synchronization is successful, or `false` if an error occurs.
 */
export declare function syncForkWithUpstream(branchName: string): Promise<void>;
export declare function getRemotes(): Promise<string[]>;
