"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.checkGithubToken = checkGithubToken;
exports.isGithubPersonalAccessToken = isGithubPersonalAccessToken;
exports.isGithubServerToServerToken = isGithubServerToServerToken;
exports.isGithubFineGrainedPersonalAccessToken = isGithubFineGrainedPersonalAccessToken;
exports.findGithubToken = findGithubToken;
exports.takePersonalAccessTokenIfPossible = takePersonalAccessTokenIfPossible;
const tslib_1 = require("tslib");
const global_1 = require("../config/global");
const logger_1 = require("../logger");
const github_release_attachments_1 = require("../modules/datasource/github-release-attachments");
const github_releases_1 = require("../modules/datasource/github-releases");
const github_tags_1 = require("../modules/datasource/github-tags");
const memCache = tslib_1.__importStar(require("../util/cache/memory"));
const hostRules = tslib_1.__importStar(require("./host-rules"));
function checkGithubToken(packageFiles = {}) {
    const { token } = hostRules.find({
        hostType: 'github',
        url: 'https://api.github.com',
    });
    if (token) {
        logger_1.logger.trace('GitHub token is found');
        return;
    }
    if (!global_1.GlobalConfig.get('githubTokenWarn')) {
        logger_1.logger.trace('GitHub token warning is disabled');
        return;
    }
    const githubDeps = [];
    const deps = Object.values(packageFiles)
        .flat()
        .map((file) => file.deps)
        .flat();
    for (const dep of deps) {
        if (!dep.skipReason &&
            (dep.datasource === github_tags_1.GithubTagsDatasource.id ||
                dep.datasource === github_releases_1.GithubReleasesDatasource.id ||
                dep.datasource === github_release_attachments_1.GithubReleaseAttachmentsDatasource.id)) {
            dep.skipReason = 'github-token-required';
            if (dep.depName) {
                githubDeps.push(dep.depName);
            }
        }
    }
    if (githubDeps.length > 0) {
        const warningLogged = memCache.get('github-token-required-warning-logged');
        if (!warningLogged) {
            const withoutDuplicates = [...new Set(githubDeps)];
            logger_1.logger.warn({ githubDeps: withoutDuplicates }, `GitHub token is required for some dependencies`);
            memCache.set('github-token-required-warning-logged', true);
        }
    }
}
function isGithubPersonalAccessToken(token) {
    return token.startsWith('ghp_');
}
function isGithubServerToServerToken(token) {
    return token.startsWith('ghs_');
}
function isGithubFineGrainedPersonalAccessToken(token) {
    return token.startsWith('github_pat_');
}
function findGithubToken(searchResult) {
    return searchResult?.token?.replace('x-access-token:', '');
}
function takePersonalAccessTokenIfPossible(githubToken, gitTagsGithubToken) {
    if (gitTagsGithubToken && isGithubPersonalAccessToken(gitTagsGithubToken)) {
        logger_1.logger.debug('Using GitHub Personal Access Token (git-tags)');
        return gitTagsGithubToken;
    }
    if (githubToken && isGithubPersonalAccessToken(githubToken)) {
        logger_1.logger.debug('Using GitHub Personal Access Token');
        return githubToken;
    }
    if (gitTagsGithubToken &&
        isGithubFineGrainedPersonalAccessToken(gitTagsGithubToken)) {
        logger_1.logger.debug('Using GitHub Fine-grained Personal Access Token (git-tags)');
        return gitTagsGithubToken;
    }
    if (githubToken && isGithubFineGrainedPersonalAccessToken(githubToken)) {
        logger_1.logger.debug('Using GitHub Fine-grained Personal Access Token');
        return githubToken;
    }
    if (gitTagsGithubToken) {
        if (isGithubServerToServerToken(gitTagsGithubToken)) {
            logger_1.logger.debug('Using GitHub Server-to-Server token (git-tags)');
        }
        else {
            logger_1.logger.debug('Using unknown GitHub token type (git-tags)');
        }
        return gitTagsGithubToken;
    }
    if (githubToken) {
        if (isGithubServerToServerToken(githubToken)) {
            logger_1.logger.debug('Using GitHub Server-to-Server token');
        }
        else {
            logger_1.logger.debug('Using unknown GitHub token type');
        }
    }
    return githubToken;
}
//# sourceMappingURL=check-token.js.map