"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RepoCacheBase = void 0;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const global_1 = require("../../../../config/global");
const logger_1 = require("../../../../logger");
const compress_1 = require("../../../compress");
const hash_1 = require("../../../hash");
const stringify_1 = require("../../../stringify");
const common_1 = require("../common");
const http_cache_1 = require("../http-cache");
const schema_1 = require("../schema");
class RepoCacheBase {
    repository;
    fingerprint;
    platform = global_1.GlobalConfig.get('platform');
    oldHash = null;
    data = {};
    constructor(repository, fingerprint) {
        this.repository = repository;
        this.fingerprint = fingerprint;
    }
    static parseData(input) {
        const data = JSON.parse(input);
        // istanbul ignore next
        if (data.branches) {
            for (const branch of data.branches) {
                if (branch.branchFingerprint) {
                    branch.commitFingerprint = branch.branchFingerprint;
                    delete branch.branchFingerprint;
                }
            }
        }
        return data;
    }
    async restore(oldCache) {
        if (oldCache.fingerprint !== this.fingerprint) {
            logger_1.logger.debug('Repository cache fingerprint is invalid');
            return;
        }
        const jsonStr = await (0, compress_1.decompressFromBase64)(oldCache.payload);
        this.data = RepoCacheBase.parseData(jsonStr);
        this.oldHash = oldCache.hash;
    }
    async load() {
        try {
            const oldCache = await this.read();
            if (!is_1.default.string(oldCache)) {
                logger_1.logger.debug(`RepoCacheBase.load() - expecting data of type 'string' received '${typeof oldCache}' instead - skipping`);
                return;
            }
            if (!is_1.default.nonEmptyString(oldCache)) {
                logger_1.logger.debug('RepoCacheBase.load() - cache file is empty - skipping');
                return;
            }
            const cacheV13 = schema_1.RepoCacheV13.safeParse(oldCache);
            if (cacheV13.success) {
                await this.restore(cacheV13.data);
                logger_1.logger.debug('Repository cache is restored from revision 13');
                return;
            }
            logger_1.logger.warn({ err: cacheV13.error }, 'Repository cache is invalid');
        }
        catch (err) /* istanbul ignore next: not easily testable */ {
            logger_1.logger.debug({ err }, 'Error reading repository cache');
        }
    }
    async save() {
        (0, http_cache_1.cleanupHttpCache)(this.data);
        const jsonStr = (0, stringify_1.safeStringify)(this.data);
        const hashedJsonStr = (0, hash_1.hash)(jsonStr);
        if (hashedJsonStr === this.oldHash) {
            return;
        }
        const revision = common_1.CACHE_REVISION;
        const repository = this.repository;
        const fingerprint = this.fingerprint;
        const payload = await (0, compress_1.compressToBase64)(jsonStr);
        await this.write({
            revision,
            repository,
            fingerprint,
            payload,
            hash: hashedJsonStr,
        });
    }
    getData() {
        return this.data;
    }
    isModified() {
        if (!this.oldHash) {
            return undefined;
        }
        const jsonStr = (0, stringify_1.safeStringify)(this.data);
        return (0, hash_1.hash)(jsonStr) !== this.oldHash;
    }
}
exports.RepoCacheBase = RepoCacheBase;
//# sourceMappingURL=base.js.map