"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.api = exports.supportedRangeStrategies = exports.supportsRanges = exports.urls = exports.displayName = exports.id = void 0;
const logger_1 = require("../../../logger");
const regex_1 = require("../../../util/regex");
const range_1 = require("./range");
const util_1 = require("./util");
exports.id = 'pvp';
exports.displayName = 'Package Versioning Policy (Haskell)';
exports.urls = ['https://pvp.haskell.org'];
exports.supportsRanges = true;
exports.supportedRangeStrategies = ['widen'];
const digitsAndDots = (0, regex_1.regEx)(/^[\d.]+$/);
function isGreaterThan(version, other) {
    const versionIntMajor = (0, util_1.extractAllParts)(version);
    const otherIntMajor = (0, util_1.extractAllParts)(other);
    if (versionIntMajor === null || otherIntMajor === null) {
        return false;
    }
    return (0, util_1.compareIntArray)(versionIntMajor, otherIntMajor) === 'gt';
}
function getMajor(version) {
    // This basically can't be implemented correctly, since
    // 1.1 and 1.10 become equal when converted to float.
    // Consumers should use isSame instead.
    const parts = (0, util_1.getParts)(version);
    if (parts === null) {
        return null;
    }
    return Number(parts.major.join('.'));
}
function getMinor(version) {
    const parts = (0, util_1.getParts)(version);
    if (parts === null || parts.minor.length === 0) {
        return null;
    }
    return Number(parts.minor.join('.'));
}
function getPatch(version) {
    const parts = (0, util_1.getParts)(version);
    if (parts === null || parts.patch.length === 0) {
        return null;
    }
    return Number(parts.patch[0] + '.' + parts.patch.slice(1).join(''));
}
function matches(version, range) {
    const parsed = (0, range_1.parseRange)(range);
    if (parsed === null) {
        return false;
    }
    const ver = (0, util_1.extractAllParts)(version);
    const lower = (0, util_1.extractAllParts)(parsed.lower);
    const upper = (0, util_1.extractAllParts)(parsed.upper);
    if (ver === null || lower === null || upper === null) {
        return false;
    }
    return ('gt' === (0, util_1.compareIntArray)(upper, ver) &&
        ['eq', 'lt'].includes((0, util_1.compareIntArray)(lower, ver)));
}
function satisfyingVersion(versions, range, reverse) {
    const copy = versions.slice(0);
    copy.sort((a, b) => {
        const multiplier = reverse ? 1 : -1;
        return sortVersions(a, b) * multiplier;
    });
    const result = copy.find((v) => matches(v, range));
    return result ?? null;
}
function getSatisfyingVersion(versions, range) {
    return satisfyingVersion(versions, range, false);
}
function minSatisfyingVersion(versions, range) {
    return satisfyingVersion(versions, range, true);
}
function isLessThanRange(version, range) {
    const parsed = (0, range_1.parseRange)(range);
    if (parsed === null) {
        return false;
    }
    const compos = (0, util_1.extractAllParts)(version);
    const lower = (0, util_1.extractAllParts)(parsed.lower);
    if (compos === null || lower === null) {
        return false;
    }
    return 'lt' === (0, util_1.compareIntArray)(compos, lower);
}
function getNewValue({ currentValue, newVersion, rangeStrategy, }) {
    if (rangeStrategy !== 'widen') {
        logger_1.logger.info({ rangeStrategy, currentValue, newVersion }, `PVP can't handle this range strategy.`);
        return null;
    }
    const parsed = (0, range_1.parseRange)(currentValue);
    if (parsed === null) {
        logger_1.logger.info({ currentValue, newVersion }, 'could not parse PVP version range');
        return null;
    }
    if (isLessThanRange(newVersion, currentValue)) {
        // ignore new releases in old release series
        return null;
    }
    if (matches(newVersion, currentValue)) {
        // the upper bound is already high enough
        return null;
    }
    const compos = (0, util_1.getParts)(newVersion);
    if (compos === null) {
        return null;
    }
    const majorPlusOne = (0, util_1.plusOne)(compos.major);
    // istanbul ignore next: since all versions that can be parsed, can also be bumped, this can never happen
    if (!matches(newVersion, `>=${parsed.lower} && <${majorPlusOne}`)) {
        logger_1.logger.warn({ newVersion }, "Even though the major bound was bumped, the newVersion still isn't accepted.");
        return null;
    }
    return `>=${parsed.lower} && <${majorPlusOne}`;
}
function isSame(type, a, b) {
    const aParts = (0, util_1.getParts)(a);
    const bParts = (0, util_1.getParts)(b);
    if (aParts === null || bParts === null) {
        return false;
    }
    if (type === 'major') {
        return 'eq' === (0, util_1.compareIntArray)(aParts.major, bParts.major);
    }
    else if (type === 'minor') {
        return 'eq' === (0, util_1.compareIntArray)(aParts.minor, bParts.minor);
    }
    else {
        return 'eq' === (0, util_1.compareIntArray)(aParts.patch, bParts.patch);
    }
}
function subset(subRange, superRange) {
    const sub = (0, range_1.parseRange)(subRange);
    const sup = (0, range_1.parseRange)(superRange);
    if (sub === null || sup === null) {
        return undefined;
    }
    const subLower = (0, util_1.extractAllParts)(sub.lower);
    const subUpper = (0, util_1.extractAllParts)(sub.upper);
    const supLower = (0, util_1.extractAllParts)(sup.lower);
    const supUpper = (0, util_1.extractAllParts)(sup.upper);
    if (subLower === null ||
        subUpper === null ||
        supLower === null ||
        supUpper === null) {
        return undefined;
    }
    if ('lt' === (0, util_1.compareIntArray)(subLower, supLower)) {
        return false;
    }
    if ('gt' === (0, util_1.compareIntArray)(subUpper, supUpper)) {
        return false;
    }
    return true;
}
function isVersion(maybeRange) {
    return typeof maybeRange === 'string' && (0, range_1.parseRange)(maybeRange) === null;
}
function isValid(ver) {
    return (0, util_1.extractAllParts)(ver) !== null || (0, range_1.parseRange)(ver) !== null;
}
function isSingleVersion(range) {
    const noSpaces = range.trim();
    return noSpaces.startsWith('==') && digitsAndDots.test(noSpaces.slice(2));
}
function equals(a, b) {
    const aParts = (0, util_1.extractAllParts)(a);
    const bParts = (0, util_1.extractAllParts)(b);
    if (aParts === null || bParts === null) {
        return false;
    }
    return 'eq' === (0, util_1.compareIntArray)(aParts, bParts);
}
function sortVersions(a, b) {
    if (equals(a, b)) {
        return 0;
    }
    return isGreaterThan(a, b) ? 1 : -1;
}
function isStable(version) {
    return true;
}
function isCompatible(version) {
    return true;
}
exports.api = {
    isValid,
    isVersion,
    isStable,
    isCompatible,
    getMajor,
    getMinor,
    getPatch,
    isSingleVersion,
    sortVersions,
    equals,
    matches,
    getSatisfyingVersion,
    minSatisfyingVersion,
    isLessThanRange,
    isGreaterThan,
    getNewValue,
    isSame,
    subset,
};
exports.default = exports.api;
//# sourceMappingURL=index.js.map