"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.api = exports.supportedRangeStrategies = exports.supportsRanges = exports.urls = exports.displayName = exports.id = void 0;
const tslib_1 = require("tslib");
const rattler_1 = require("@baszalmstra/rattler");
const pep440 = tslib_1.__importStar(require("../pep440"));
function parse(v) {
    try {
        return new rattler_1.Version(v);
    }
    catch {
        return null;
    }
}
exports.id = 'conda';
exports.displayName = 'conda';
exports.urls = [
    'https://docs.conda.io/projects/conda-build/en/stable/resources/package-spec.html#package-match-specifications',
];
exports.supportsRanges = true;
exports.supportedRangeStrategies = [
    'bump',
    'widen',
    'pin',
    'replace',
];
function isValidVersion(s) {
    try {
        new rattler_1.Version(s);
        return true;
    }
    catch {
        return false;
    }
}
function isValidVersionSpec(s) {
    try {
        new rattler_1.VersionSpec(s);
        return true;
    }
    catch {
        return false;
    }
}
function isValid(input) {
    return isValidVersion(input) || isValidVersionSpec(input);
}
function matches(version, range) {
    try {
        return new rattler_1.VersionSpec(range).matches(new rattler_1.Version(version));
    }
    catch {
        return false;
    }
}
function isSingleVersion(input) {
    if (!input.startsWith('=')) {
        return false;
    }
    return isValidVersion(input.replace(/^==/, '').trimStart());
}
function getNewValue({ currentValue, rangeStrategy, currentVersion, newVersion, isReplacement, }) {
    if (rangeStrategy === 'pin') {
        return '==' + newVersion;
    }
    if (currentValue === '*') {
        if (rangeStrategy === 'bump') {
            return '>=' + newVersion;
        }
        // don't think you can widen or replace `*`
        return null;
    }
    const normalizedCurrentValue = new rattler_1.VersionSpec(currentValue).toString();
    // it's valid range spec in conda to write `3.12.*`, translate to pep440 `==3.12.*`
    if (/^(\d+\.)+\*$/.test(normalizedCurrentValue)) {
        const newValue = pep440.api.getNewValue({
            currentValue: '==' + normalizedCurrentValue,
            rangeStrategy,
            currentVersion,
            newVersion,
            isReplacement,
        });
        return newValue?.replace(/^==/, '') ?? null;
    }
    return pep440.api.getNewValue({
        currentValue: normalizedCurrentValue,
        rangeStrategy,
        currentVersion,
        newVersion,
        isReplacement,
    });
}
function sortVersions(version, other) {
    return new rattler_1.Version(version).compare(new rattler_1.Version(other));
}
function equals(version, other) {
    const v2 = parse(other);
    if (!v2) {
        return false;
    }
    return parse(version)?.equals(v2) ?? false;
}
function isStable(version) {
    return !(parse(version)?.isDev ?? true);
}
function isCompatible(version, current) {
    return true;
}
function getMajor(version) {
    return parse(version)?.asMajorMinor?.()?.[0] ?? null;
}
function getMinor(version) {
    return parse(version)?.asMajorMinor?.()?.[1] ?? null;
}
function getPatch(version) {
    try {
        return pep440.api.getPatch(version);
    }
    catch {
        return null;
    }
}
function isGreaterThan(version, other) {
    return sortVersions(version, other) > 0;
}
function getSatisfyingVersion(versions, range) {
    const spec = new rattler_1.VersionSpec(range);
    const satisfiedVersions = versions
        .map((v) => {
        return [new rattler_1.Version(v), v];
    })
        .filter(([v, raw]) => spec.matches(v))
        .sort((a, b) => {
        return a[0].compare(b[0]);
    });
    if (satisfiedVersions.length === 0) {
        return null;
    }
    return satisfiedVersions[satisfiedVersions.length - 1][1];
}
function minSatisfyingVersion(versions, range) {
    const spec = new rattler_1.VersionSpec(range);
    const satisfiedVersions = versions
        .map((v) => {
        return [new rattler_1.Version(v), v];
    })
        .filter(([v, raw]) => spec.matches(v))
        .sort((a, b) => {
        return a[0].compare(b[0]);
    });
    if (satisfiedVersions.length === 0) {
        return null;
    }
    return satisfiedVersions[0][1];
}
exports.api = {
    equals,
    isValid,
    isVersion: isValidVersion,
    isSingleVersion,
    // conda doesn't have stable version and non-stable version
    // for example, tzdata has version 2024a but it's stable
    isStable,
    // conda use version are always compatible with each other.
    isCompatible,
    getMajor,
    getMinor,
    // sadly conda version doesn't have a concept of patch version
    // so we try to use pep440 get a patch version.
    getPatch,
    isGreaterThan,
    getSatisfyingVersion,
    minSatisfyingVersion,
    getNewValue,
    matches,
    sortVersions,
};
//# sourceMappingURL=index.js.map