"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitlabPrCache = void 0;
const tslib_1 = require("tslib");
const dequal_1 = require("dequal");
const luxon_1 = require("luxon");
const logger_1 = require("../../../logger");
const memCache = tslib_1.__importStar(require("../../../util/cache/memory"));
const repository_1 = require("../../../util/cache/repository");
const url_1 = require("../../../util/url");
const utils_1 = require("./utils");
class GitlabPrCache {
    repo;
    ignorePrAuthor;
    items = [];
    cache;
    constructor(repo, author, ignorePrAuthor) {
        this.repo = repo;
        this.ignorePrAuthor = ignorePrAuthor;
        const repoCache = (0, repository_1.getCache)();
        repoCache.platform ??= {};
        repoCache.platform.gitlab ??= {};
        let pullRequestCache = repoCache.platform.gitlab
            .pullRequestsCache;
        if (!pullRequestCache) {
            logger_1.logger.debug('Initializing new PR cache at repository cache');
            pullRequestCache = {
                items: {},
                updated_at: null,
                author,
            };
        }
        else if (pullRequestCache.author !== author) {
            logger_1.logger.debug('Resetting PR cache because authors do not match');
            pullRequestCache = {
                items: {},
                updated_at: null,
                author,
            };
        }
        repoCache.platform.gitlab.pullRequestsCache = pullRequestCache;
        this.cache = pullRequestCache;
        this.updateItems();
    }
    static async init(http, repo, author, ignorePrAuthor) {
        const res = new GitlabPrCache(repo, author, ignorePrAuthor);
        const isSynced = memCache.get('gitlab-pr-cache-synced');
        if (!isSynced) {
            await res.sync(http);
            memCache.set('gitlab-pr-cache-synced', true);
        }
        return res;
    }
    getPrs() {
        return this.items;
    }
    static async getPrs(http, repo, author, ignorePrAuthor) {
        const prCache = await GitlabPrCache.init(http, repo, author, ignorePrAuthor);
        return prCache.getPrs();
    }
    setPr(pr) {
        logger_1.logger.debug(`Adding PR #${pr.number} to the PR cache`);
        this.cache.items[pr.number] = pr;
        this.updateItems();
    }
    static async setPr(http, repo, author, item, ignorePrAuthor) {
        const prCache = await GitlabPrCache.init(http, repo, author, ignorePrAuthor);
        prCache.setPr(item);
    }
    reconcile(rawItems) {
        const { items: oldItems } = this.cache;
        let { updated_at } = this.cache;
        let needNextPage = true;
        for (const rawItem of rawItems) {
            const id = rawItem.iid;
            const oldItem = oldItems[id];
            const newItem = (0, utils_1.prInfo)(rawItem);
            const itemNewTime = luxon_1.DateTime.fromISO(rawItem.updated_at);
            if ((0, dequal_1.dequal)(oldItem, newItem)) {
                needNextPage = false;
                continue;
            }
            oldItems[id] = newItem;
            const cacheOldTime = updated_at ? luxon_1.DateTime.fromISO(updated_at) : null;
            if (!cacheOldTime || itemNewTime > cacheOldTime) {
                updated_at = rawItem.updated_at;
            }
        }
        this.cache.updated_at = updated_at;
        return needNextPage;
    }
    async sync(http) {
        logger_1.logger.debug('Syncing PR list');
        const searchParams = {
            per_page: this.items.length ? '20' : '100',
        };
        if (!this.ignorePrAuthor) {
            searchParams.scope = 'created_by_me';
        }
        let query = (0, url_1.getQueryString)(searchParams);
        while (query) {
            const res = await http.getJsonUnchecked(`/projects/${this.repo}/merge_requests?${query}`, {
                memCache: false,
            });
            const needNextPage = this.reconcile(res.body);
            if (!needNextPage) {
                break;
            }
            const uri = (0, url_1.parseUrl)((0, url_1.parseLinkHeader)(res.headers.link)?.next?.url);
            query = uri ? uri.search : null;
        }
        this.updateItems();
        return this;
    }
    /**
     * Ensure the pr cache starts with the most recent PRs.
     * JavaScript ensures that the cache is sorted by PR number.
     */
    updateItems() {
        this.items = Object.values(this.cache.items).reverse();
    }
}
exports.GitlabPrCache = GitlabPrCache;
//# sourceMappingURL=pr-cache.js.map