"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GithubIssueCache = exports.GithubIssue = void 0;
const tslib_1 = require("tslib");
const luxon_1 = require("luxon");
const zod_1 = require("zod");
const logger_1 = require("../../../logger");
const memCache = tslib_1.__importStar(require("../../../util/cache/memory"));
const repository_1 = require("../../../util/cache/repository");
const GithubIssueBase = zod_1.z.object({
    number: zod_1.z.number(),
    state: zod_1.z.string().transform((val) => val.toLowerCase()),
    title: zod_1.z.string(),
    body: zod_1.z.string(),
});
const GithubGraphqlIssue = GithubIssueBase.extend({
    updatedAt: zod_1.z.string(),
}).transform((issue) => {
    const lastModified = issue.updatedAt;
    const { number, state, title, body } = issue;
    return { number, state, title, body, lastModified };
});
const GithubRestIssue = GithubIssueBase.extend({
    updated_at: zod_1.z.string(),
}).transform((issue) => {
    const lastModified = issue.updated_at;
    const { number, state, title, body } = issue;
    return { number, state, title, body, lastModified };
});
exports.GithubIssue = zod_1.z.union([GithubGraphqlIssue, GithubRestIssue]);
class GithubIssueCache {
    static reset(cacheData) {
        memCache.set('github-issues-reconcile-queue', null);
        const repoCache = (0, repository_1.getCache)();
        repoCache.platform ??= {};
        repoCache.platform.github ??= {};
        if (cacheData) {
            repoCache.platform.github.issuesCache = cacheData;
        }
        else {
            delete repoCache.platform.github.issuesCache;
        }
    }
    static get data() {
        let cacheData = (0, repository_1.getCache)().platform?.github
            ?.issuesCache;
        if (!cacheData) {
            return null;
        }
        cacheData = this.reconcile(cacheData);
        return cacheData;
    }
    static getIssues() {
        const cacheData = this.data;
        if (!cacheData) {
            return null;
        }
        const sortedResult = Object.values(cacheData).sort(({ lastModified: a }, { lastModified: b }) => luxon_1.DateTime.fromISO(b).toMillis() - luxon_1.DateTime.fromISO(a).toMillis());
        return sortedResult;
    }
    static setIssues(issues) {
        const cacheData = {};
        for (const issue of issues) {
            cacheData[issue.number] = issue;
        }
        logger_1.logger.debug(`Issues cache: Setting ${issues.length} issues in cache`);
        this.reset(cacheData);
    }
    static updateIssue(issue) {
        const cacheData = this.data;
        if (cacheData) {
            logger_1.logger.debug(`Issues cache: Updating issue ${issue.number} in cache`);
            cacheData[issue.number] = issue;
        }
    }
    static deleteIssue(number) {
        const cacheData = this.data;
        if (cacheData) {
            logger_1.logger.debug(`Issues cache: Deleting issue ${number} from cache`);
            delete cacheData[number];
        }
    }
    /**
     * At the moment of repo initialization, repository cache is not available.
     * What we can do is to store issues for later reconciliation.
     */
    static addIssuesToReconcile(issues) {
        logger_1.logger.debug(`Issues cache: Adding ${issues?.length} issues to reconcile queue`);
        memCache.set('github-issues-reconcile-queue', issues);
    }
    static reconcile(cacheData) {
        const issuesToReconcile = memCache.get('github-issues-reconcile-queue');
        if (!issuesToReconcile) {
            return cacheData;
        }
        let isReconciled = false;
        for (const issue of issuesToReconcile) {
            const cachedIssue = cacheData[issue.number];
            // If we reached the item which is already in the cache,
            // it means sync is done.
            if (cachedIssue &&
                cachedIssue.number === issue.number &&
                cachedIssue.lastModified === issue.lastModified) {
                isReconciled = true;
                logger_1.logger.debug(`Issues cache: Done reconciling at issue ${issue.number}`);
                break;
            }
            cacheData[issue.number] = issue;
        }
        // If we've just iterated over all the items in the cache,
        // it means sync is also done.
        if (issuesToReconcile.length >= Object.keys(cacheData).length) {
            logger_1.logger.debug(`Issues cache: Done reconciling by iterating over all items`);
            isReconciled = true;
        }
        if (!isReconciled) {
            logger_1.logger.debug('Issues cache: reset');
            this.reset(null);
            return null;
        }
        logger_1.logger.debug('Issues cache: synced');
        this.reset(cacheData);
        return cacheData;
    }
}
exports.GithubIssueCache = GithubIssueCache;
//# sourceMappingURL=issue.js.map