"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiCache = void 0;
const dequal_1 = require("dequal");
const luxon_1 = require("luxon");
const logger_1 = require("../../../logger");
class ApiCache {
    cache;
    constructor(cache) {
        this.cache = cache;
    }
    getItems() {
        const items = Object.values(this.cache.items);
        return items;
    }
    getItem(number) {
        return this.cache.items[number] ?? null;
    }
    /**
     * It intentionally doesn't alter `lastModified` cache field.
     *
     * The point is to allow cache modifications during run, but
     * force fetching and refreshing of modified items next run.
     */
    updateItem(item) {
        this.cache.items[item.number] = item;
    }
    /**
     * Copies items from `page` to `cache`.
     * Updates internal cache timestamp.
     *
     * @param cache Cache object
     * @param page List of cacheable items, sorted by `updated_at` field
     * starting from the most recently updated.
     * @returns `true` when the next page is likely to contain fresh items,
     * otherwise `false`.
     */
    reconcile(page) {
        const { items } = this.cache;
        let { lastModified } = this.cache;
        let needNextPage = true;
        for (const newItem of page) {
            const number = newItem.number;
            const oldItem = items[number];
            const itemNewTime = luxon_1.DateTime.fromISO(newItem.updated_at);
            const itemOldTime = oldItem?.updated_at
                ? luxon_1.DateTime.fromISO(oldItem.updated_at)
                : null;
            if (!(0, dequal_1.dequal)(oldItem, newItem)) {
                logger_1.logger.trace(`PR cache: updating item ${number}`);
                items[number] = newItem;
            }
            needNextPage = itemOldTime ? itemOldTime < itemNewTime : true;
            const cacheOldTime = lastModified ? luxon_1.DateTime.fromISO(lastModified) : null;
            if (!cacheOldTime || itemNewTime > cacheOldTime) {
                lastModified = newItem.updated_at;
            }
        }
        this.cache.lastModified = lastModified;
        return needNextPage;
    }
}
exports.ApiCache = ApiCache;
//# sourceMappingURL=api-cache.js.map