"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DRAFT_PREFIX = exports.API_PATH = void 0;
exports.smartLinks = smartLinks;
exports.trimTrailingApiPath = trimTrailingApiPath;
exports.getRepoUrl = getRepoUrl;
exports.getMergeMethod = getMergeMethod;
exports.toRenovatePR = toRenovatePR;
exports.usableRepo = usableRepo;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const error_messages_1 = require("../../../constants/error-messages");
const logger_1 = require("../../../logger");
const hostRules = tslib_1.__importStar(require("../../../util/host-rules"));
const regex_1 = require("../../../util/regex");
const url_1 = require("../../../util/url");
const pr_body_1 = require("../pr-body");
function smartLinks(body) {
    return body?.replace((0, regex_1.regEx)(/\]\(\.\.\/pull\//g), '](pulls/');
}
function trimTrailingApiPath(url) {
    return url?.replace((0, regex_1.regEx)(/api\/v1\/?$/g), '');
}
function getRepoUrl(repo, gitUrl, endpoint) {
    if (gitUrl === 'ssh') {
        if (!repo.ssh_url) {
            throw new Error(error_messages_1.CONFIG_GIT_URL_UNAVAILABLE);
        }
        logger_1.logger.debug(`Using SSH URL: ${repo.ssh_url}`);
        return repo.ssh_url;
    }
    // Find options for current host and determine Git endpoint
    const opts = hostRules.find({
        hostType: 'gitea',
        url: endpoint,
    });
    if (gitUrl === 'endpoint') {
        const url = (0, url_1.parseUrl)(endpoint);
        if (!url) {
            throw new Error(error_messages_1.CONFIG_GIT_URL_UNAVAILABLE);
        }
        url.username = opts.token ?? '';
        url.pathname = `${url.pathname}${repo.full_name}.git`;
        logger_1.logger.debug({ url: url.toString() }, 'using URL based on configured endpoint');
        return url.toString();
    }
    if (!repo.clone_url) {
        throw new Error(error_messages_1.CONFIG_GIT_URL_UNAVAILABLE);
    }
    logger_1.logger.debug(`Using HTTP URL: ${repo.clone_url}`);
    const repoUrl = (0, url_1.parseUrl)(repo.clone_url);
    if (!repoUrl) {
        throw new Error(error_messages_1.CONFIG_GIT_URL_UNAVAILABLE);
    }
    repoUrl.username = opts.token ?? '';
    return repoUrl.toString();
}
function getMergeMethod(strategy) {
    switch (strategy) {
        case 'fast-forward':
            return 'rebase';
        case 'merge-commit':
            return 'merge';
        case 'rebase':
            return 'rebase-merge';
        case 'squash':
            return strategy;
        case 'auto':
        default:
            return null;
    }
}
exports.API_PATH = '/api/v1';
exports.DRAFT_PREFIX = 'WIP: ';
const reconfigurePrRegex = (0, regex_1.regEx)(/reconfigure$/g);
function toRenovatePR(data, author) {
    if (!data) {
        return null;
    }
    if (!data.base?.ref ||
        !data.head?.label ||
        !data.head?.sha ||
        !data.head?.repo?.full_name) {
        logger_1.logger.trace(`Skipping Pull Request #${data.number} due to missing base and/or head branch`);
        return null;
    }
    const createdBy = data.user?.username;
    if (createdBy &&
        author &&
        !reconfigurePrRegex.test(data.head.label) &&
        createdBy !== author) {
        return null;
    }
    let title = data.title;
    let isDraft = false;
    if (title.startsWith(exports.DRAFT_PREFIX)) {
        title = title.substring(exports.DRAFT_PREFIX.length);
        isDraft = true;
    }
    const labels = (data?.labels ?? []).map((l) => l.name);
    return {
        labels,
        number: data.number,
        state: data.merged ? 'merged' : data.state,
        title,
        isDraft,
        bodyStruct: (0, pr_body_1.getPrBodyStruct)(data.body),
        sha: data.head.sha,
        sourceBranch: data.head.label,
        targetBranch: data.base.ref,
        sourceRepo: data.head.repo.full_name,
        createdAt: data.created_at,
        cannotMergeReason: data.mergeable
            ? undefined
            : `pr.mergeable="${data.mergeable}"`,
        hasAssignees: !!(data.assignee?.login ?? is_1.default.nonEmptyArray(data.assignees)),
    };
}
/**
 * Check if a repository is usable.
 * A repo isn't usable if one of the following conditions is met:
 * - The repo is a `mirror`
 * - We don't have pull or push permissions
 * - Pull requests are disabled
 * @param repo Repo to check
 * @returns  `true` if the repository is usable, `false` otherwise
 */
function usableRepo(repo) {
    if (repo.mirror === true) {
        return false;
    }
    if (repo.permissions.pull === false || repo.permissions.push === false) {
        logger_1.logger.debug(`Skipping repository ${repo.full_name} because of missing pull or push permissions`);
        return false;
    }
    if (repo.has_pull_requests === false) {
        logger_1.logger.debug(`Skipping repository ${repo.full_name} because pull requests are disabled`);
        return false;
    }
    return true;
}
//# sourceMappingURL=utils.js.map