"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.renovateToGiteaStatusMapping = exports.giteaToRenovateStatusMapping = exports.giteaHttp = void 0;
exports.getCurrentUser = getCurrentUser;
exports.getVersion = getVersion;
exports.searchRepos = searchRepos;
exports.orgListRepos = orgListRepos;
exports.getRepo = getRepo;
exports.getRepoContents = getRepoContents;
exports.createPR = createPR;
exports.updatePR = updatePR;
exports.closePR = closePR;
exports.mergePR = mergePR;
exports.getPR = getPR;
exports.getPRByBranch = getPRByBranch;
exports.requestPrReviewers = requestPrReviewers;
exports.createIssue = createIssue;
exports.updateIssue = updateIssue;
exports.updateIssueLabels = updateIssueLabels;
exports.closeIssue = closeIssue;
exports.searchIssues = searchIssues;
exports.getIssue = getIssue;
exports.getRepoLabels = getRepoLabels;
exports.getOrgLabels = getOrgLabels;
exports.unassignLabel = unassignLabel;
exports.createComment = createComment;
exports.updateComment = updateComment;
exports.deleteComment = deleteComment;
exports.getComments = getComments;
exports.createCommitStatus = createCommitStatus;
exports.getCombinedCommitStatus = getCombinedCommitStatus;
exports.getBranch = getBranch;
const logger_1 = require("../../../logger");
const gitea_1 = require("../../../util/http/gitea");
const url_1 = require("../../../util/url");
const utils_1 = require("./utils");
exports.giteaHttp = new gitea_1.GiteaHttp();
const urlEscape = (raw) => encodeURIComponent(raw);
const commitStatusStates = [
    'unknown',
    'success',
    'pending',
    'warning',
    'failure',
    'error',
];
async function getCurrentUser(options) {
    const url = `${utils_1.API_PATH}/user`;
    const res = await exports.giteaHttp.getJsonUnchecked(url, options);
    return res.body;
}
async function getVersion(options) {
    const url = `${utils_1.API_PATH}/version`;
    const res = await exports.giteaHttp.getJsonUnchecked(url, options);
    return res.body.version;
}
async function searchRepos(params, options) {
    const query = (0, url_1.getQueryString)(params);
    const url = `${utils_1.API_PATH}/repos/search?${query}`;
    const res = await exports.giteaHttp.getJsonUnchecked(url, {
        ...options,
        paginate: true,
    });
    if (!res.body.ok) {
        throw new Error('Unable to search for repositories, ok flag has not been set');
    }
    return res.body.data;
}
async function orgListRepos(organization, options) {
    const url = `${utils_1.API_PATH}/orgs/${organization}/repos`;
    const res = await exports.giteaHttp.getJsonUnchecked(url, {
        ...options,
        paginate: true,
    });
    return res.body;
}
async function getRepo(repoPath, options) {
    const url = `${utils_1.API_PATH}/repos/${repoPath}`;
    const res = await exports.giteaHttp.getJsonUnchecked(url, options);
    return res.body;
}
async function getRepoContents(repoPath, filePath, ref, options) {
    const query = (0, url_1.getQueryString)(ref ? { ref } : {});
    const url = `${utils_1.API_PATH}/repos/${repoPath}/contents/${urlEscape(filePath)}?${query}`;
    const res = await exports.giteaHttp.getJsonUnchecked(url, options);
    if (res.body.content) {
        res.body.contentString = Buffer.from(res.body.content, 'base64').toString();
    }
    return res.body;
}
async function createPR(repoPath, params, options) {
    const url = `${utils_1.API_PATH}/repos/${repoPath}/pulls`;
    const res = await exports.giteaHttp.postJson(url, {
        ...options,
        body: params,
    });
    return res.body;
}
async function updatePR(repoPath, idx, params, options) {
    const url = `${utils_1.API_PATH}/repos/${repoPath}/pulls/${idx}`;
    const res = await exports.giteaHttp.patchJson(url, {
        ...options,
        body: params,
    });
    return res.body;
}
async function closePR(repoPath, idx, options) {
    await updatePR(repoPath, idx, {
        ...options,
        state: 'closed',
    });
}
async function mergePR(repoPath, idx, params, options) {
    const url = `${utils_1.API_PATH}/repos/${repoPath}/pulls/${idx}/merge`;
    await exports.giteaHttp.postJson(url, {
        ...options,
        body: params,
    });
}
async function getPR(repoPath, idx, options) {
    const url = `${utils_1.API_PATH}/repos/${repoPath}/pulls/${idx}`;
    const res = await exports.giteaHttp.getJsonUnchecked(url, options);
    return res.body;
}
async function getPRByBranch(repoPath, base, head, options) {
    const url = `${utils_1.API_PATH}/repos/${repoPath}/pulls/${base}/${head}`;
    try {
        const res = await exports.giteaHttp.getJsonUnchecked(url, options);
        return res.body;
    }
    catch (err) {
        logger_1.logger.trace({ err }, 'Error while fetching PR');
        if (err.statusCode !== 404) {
            logger_1.logger.debug({ err }, 'Error while fetching PR');
        }
        return null;
    }
}
async function requestPrReviewers(repoPath, idx, params, options) {
    const url = `${utils_1.API_PATH}/repos/${repoPath}/pulls/${idx}/requested_reviewers`;
    await exports.giteaHttp.postJson(url, {
        ...options,
        body: params,
    });
}
async function createIssue(repoPath, params, options) {
    const url = `${utils_1.API_PATH}/repos/${repoPath}/issues`;
    const res = await exports.giteaHttp.postJson(url, {
        ...options,
        body: params,
    });
    return res.body;
}
async function updateIssue(repoPath, idx, params, options) {
    const url = `${utils_1.API_PATH}/repos/${repoPath}/issues/${idx}`;
    const res = await exports.giteaHttp.patchJson(url, {
        ...options,
        body: params,
    });
    return res.body;
}
async function updateIssueLabels(repoPath, idx, params, options) {
    const url = `${utils_1.API_PATH}/repos/${repoPath}/issues/${idx}/labels`;
    const res = await exports.giteaHttp.putJson(url, {
        ...options,
        body: params,
    });
    return res.body;
}
async function closeIssue(repoPath, idx, options) {
    await updateIssue(repoPath, idx, {
        ...options,
        state: 'closed',
    });
}
async function searchIssues(repoPath, params, options) {
    const query = (0, url_1.getQueryString)({ ...params, type: 'issues' });
    const url = `${utils_1.API_PATH}/repos/${repoPath}/issues?${query}`;
    const res = await exports.giteaHttp.getJsonUnchecked(url, {
        ...options,
        paginate: true,
    });
    return res.body;
}
async function getIssue(repoPath, idx, options) {
    const url = `${utils_1.API_PATH}/repos/${repoPath}/issues/${idx}`;
    const res = await exports.giteaHttp.getJsonUnchecked(url, options);
    return res.body;
}
async function getRepoLabels(repoPath, options) {
    const url = `${utils_1.API_PATH}/repos/${repoPath}/labels`;
    const res = await exports.giteaHttp.getJsonUnchecked(url, options);
    return res.body;
}
async function getOrgLabels(orgName, options) {
    const url = `${utils_1.API_PATH}/orgs/${orgName}/labels`;
    const res = await exports.giteaHttp.getJsonUnchecked(url, options);
    return res.body;
}
async function unassignLabel(repoPath, issue, label, options) {
    const url = `${utils_1.API_PATH}/repos/${repoPath}/issues/${issue}/labels/${label}`;
    await exports.giteaHttp.deleteJson(url, options);
}
async function createComment(repoPath, issue, body, options) {
    const params = { body };
    const url = `${utils_1.API_PATH}/repos/${repoPath}/issues/${issue}/comments`;
    const res = await exports.giteaHttp.postJson(url, {
        ...options,
        body: params,
    });
    return res.body;
}
async function updateComment(repoPath, idx, body, options) {
    const params = { body };
    const url = `${utils_1.API_PATH}/repos/${repoPath}/issues/comments/${idx}`;
    const res = await exports.giteaHttp.patchJson(url, {
        ...options,
        body: params,
    });
    return res.body;
}
async function deleteComment(repoPath, idx, options) {
    const url = `${utils_1.API_PATH}/repos/${repoPath}/issues/comments/${idx}`;
    await exports.giteaHttp.deleteJson(url, options);
}
async function getComments(repoPath, issue, options) {
    const url = `${utils_1.API_PATH}/repos/${repoPath}/issues/${issue}/comments`;
    const res = await exports.giteaHttp.getJsonUnchecked(url, options);
    return res.body;
}
async function createCommitStatus(repoPath, branchCommit, params, options) {
    const url = `${utils_1.API_PATH}/repos/${repoPath}/statuses/${branchCommit}`;
    const res = await exports.giteaHttp.postJson(url, {
        ...options,
        body: params,
    });
    return res.body;
}
exports.giteaToRenovateStatusMapping = {
    unknown: 'yellow',
    success: 'green',
    pending: 'yellow',
    warning: 'red',
    failure: 'red',
    error: 'red',
};
exports.renovateToGiteaStatusMapping = {
    green: 'success',
    yellow: 'pending',
    red: 'failure',
};
function filterStatus(data) {
    const ret = {};
    for (const i of data) {
        if (!ret[i.context] || ret[i.context].id < i.id) {
            ret[i.context] = i;
        }
    }
    return Object.values(ret);
}
async function getCombinedCommitStatus(repoPath, branchName, options) {
    const url = `${utils_1.API_PATH}/repos/${repoPath}/commits/${urlEscape(branchName)}/statuses`;
    const res = await exports.giteaHttp.getJsonUnchecked(url, {
        ...options,
        paginate: true,
    });
    let worstState = 0;
    const statuses = filterStatus(res.body);
    for (const cs of statuses) {
        worstState = Math.max(worstState, commitStatusStates.indexOf(cs.status));
    }
    return {
        worstStatus: commitStatusStates[worstState],
        statuses,
    };
}
async function getBranch(repoPath, branchName, options) {
    const url = `${utils_1.API_PATH}/repos/${repoPath}/branches/${urlEscape(branchName)}`;
    const res = await exports.giteaHttp.getJsonUnchecked(url, options);
    return res.body;
}
//# sourceMappingURL=gitea-helper.js.map