"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.client = void 0;
const error_messages_1 = require("../../../constants/error-messages");
const logger_1 = require("../../../logger");
const memory_http_cache_provider_1 = require("../../../util/http/cache/memory-http-cache-provider");
const gerrit_1 = require("../../../util/http/gerrit");
const url_1 = require("../../../util/url");
const utils_1 = require("./utils");
class GerritClient {
    gerritHttp = new gerrit_1.GerritHttp();
    async getRepos() {
        const res = await this.gerritHttp.getJsonUnchecked('a/projects/?type=CODE&state=ACTIVE');
        return Object.keys(res.body);
    }
    async getProjectInfo(repository) {
        const projectInfo = await this.gerritHttp.getJsonUnchecked(`a/projects/${encodeURIComponent(repository)}`);
        if (projectInfo.body.state !== 'ACTIVE') {
            throw new Error(error_messages_1.REPOSITORY_ARCHIVED);
        }
        return projectInfo.body;
    }
    async getBranchInfo(repository) {
        const branchInfo = await this.gerritHttp.getJsonUnchecked(`a/projects/${encodeURIComponent(repository)}/branches/HEAD`);
        return branchInfo.body;
    }
    async findChanges(repository, findPRConfig) {
        /* v8 ignore start: temporary code */
        // Disables memCache (enabled by default) to be replaced by memCacheProvider
        const opts = { memCache: false };
        // TODO: should refresh the cache rather than just ignore it
        if (!findPRConfig.refreshCache) {
            opts.cacheProvider = memory_http_cache_provider_1.memCacheProvider;
        }
        /* v8 ignore stop */
        const query = {};
        if (findPRConfig.requestDetails) {
            query.o = findPRConfig.requestDetails;
        }
        if (findPRConfig.limit) {
            query.n = findPRConfig.limit;
        }
        else {
            // TODO: handle pagination instead
            query['no-limit'] = true;
        }
        const filters = GerritClient.buildSearchFilters(repository, findPRConfig);
        const queryString = `q=${filters.join('+')}&${(0, url_1.getQueryString)(query)}`;
        const changes = await this.gerritHttp.getJsonUnchecked(`a/changes/?${queryString}`, opts);
        logger_1.logger.trace(`findChanges(${queryString}) => ${changes.body.length}`);
        return changes.body;
    }
    async getChange(changeNumber, refreshCache, requestDetails) {
        /* v8 ignore start: temporary code */
        // Disables memCache (enabled by default) to be replaced by memCacheProvider
        const opts = { memCache: false };
        // TODO: should refresh the cache rather than just ignore it
        if (!refreshCache) {
            opts.cacheProvider = memory_http_cache_provider_1.memCacheProvider;
        }
        /* v8 ignore stop */
        const queryString = (0, url_1.getQueryString)({ o: requestDetails });
        const changes = await this.gerritHttp.getJsonUnchecked(`a/changes/${changeNumber}?${queryString}`, opts);
        return changes.body;
    }
    async getMergeableInfo(change) {
        const mergeable = await this.gerritHttp.getJsonUnchecked(`a/changes/${change._number}/revisions/current/mergeable`);
        return mergeable.body;
    }
    async abandonChange(changeNumber, message) {
        await this.gerritHttp.postJson(`a/changes/${changeNumber}/abandon`, {
            body: {
                message,
                notify: 'OWNER_REVIEWERS', // Avoids notifying cc's
            },
        });
    }
    async submitChange(changeNumber) {
        const change = await this.gerritHttp.postJson(`a/changes/${changeNumber}/submit`);
        return change.body;
    }
    async getMessages(changeNumber) {
        const messages = await this.gerritHttp.getJsonUnchecked(`a/changes/${changeNumber}/messages`, { memCache: false });
        return messages.body;
    }
    async addMessage(changeNumber, fullMessage, tag) {
        const message = this.normalizeMessage(fullMessage);
        await this.gerritHttp.postJson(`a/changes/${changeNumber}/revisions/current/review`, { body: { message, tag, notify: 'NONE' } });
    }
    async checkForExistingMessage(changeNumber, newMessage, msgType, messages) {
        const messagesToSearch = messages ?? (await this.getMessages(changeNumber));
        return messagesToSearch.some((existingMsg) => (msgType === undefined || msgType === existingMsg.tag) &&
            existingMsg.message.includes(newMessage));
    }
    async addMessageIfNotAlreadyExists(changeNumber, message, tag, messages) {
        const newMsg = this.normalizeMessage(message);
        if (!(await this.checkForExistingMessage(changeNumber, newMsg, tag, messages))) {
            await this.addMessage(changeNumber, newMsg, tag);
        }
    }
    async setLabel(changeNumber, label, value) {
        await this.gerritHttp.postJson(`a/changes/${changeNumber}/revisions/current/review`, { body: { labels: { [label]: value }, notify: 'NONE' } });
    }
    async deleteHashtag(changeNumber, hashtag) {
        await this.gerritHttp.postJson(`a/changes/${changeNumber}/hashtags`, {
            body: { remove: [hashtag] },
        });
    }
    async addReviewers(changeNumber, reviewers) {
        await this.gerritHttp.postJson(`a/changes/${changeNumber}/revisions/current/review`, {
            body: {
                reviewers: reviewers.map((r) => ({ reviewer: r })),
                notify: 'OWNER_REVIEWERS', // Avoids notifying cc's
            },
        });
    }
    async addAssignee(changeNumber, assignee) {
        await this.gerritHttp.putJson(
        // TODO: refactor this as this API removed in Gerrit 3.8
        `a/changes/${changeNumber}/assignee`, {
            body: { assignee },
        });
    }
    async getFile(repo, branch, fileName) {
        const base64Content = await this.gerritHttp.getText(`a/projects/${encodeURIComponent(repo)}/branches/${encodeURIComponent(branch)}/files/${encodeURIComponent(fileName)}/content`);
        return Buffer.from(base64Content.body, 'base64').toString();
    }
    normalizeMessage(message) {
        //the last \n was removed from gerrit after the comment was added...
        return message.substring(0, 0x4000).trim();
    }
    static buildSearchFilters(repository, searchConfig) {
        const filters = [
            'owner:self',
            `project:${repository}`,
            '-is:wip',
            '-is:private',
        ];
        const filterState = (0, utils_1.mapPrStateToGerritFilter)(searchConfig.state);
        if (filterState) {
            filters.push(filterState);
        }
        if (searchConfig.branchName) {
            filters.push(`footer:Renovate-Branch=${searchConfig.branchName}`);
        }
        // TODO: Use Gerrit 3.6+ hasfooter:Renovate-Branch when branchName is empty:
        //   https://gerrit-review.googlesource.com/c/gerrit/+/329488
        if (searchConfig.targetBranch) {
            filters.push(`branch:${searchConfig.targetBranch}`);
        }
        if (searchConfig.label) {
            filters.push(`label:Code-Review=${searchConfig.label}`);
        }
        if (searchConfig.prTitle) {
            // Quotes in the commit message must be escaped with a backslash:
            //   https://gerrit-review.googlesource.com/Documentation/user-search.html#search-operators
            // TODO: Use Gerrit 3.8+ subject query instead:
            //   https://gerrit-review.googlesource.com/c/gerrit/+/354037
            filters.push(`message:${encodeURIComponent('"' + searchConfig.prTitle.replaceAll('"', '\\"') + '"')}`);
        }
        return filters;
    }
}
exports.client = new GerritClient();
//# sourceMappingURL=client.js.map