"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UnresolvedPrTasks = exports.Repositories = exports.RepoInfo = exports.PagedSourceResultsSchema = void 0;
const zod_1 = require("zod");
const logger_1 = require("../../../logger");
const schema_utils_1 = require("../../../util/schema-utils");
const BitbucketSourceTypeSchema = zod_1.z.enum(['commit_directory', 'commit_file']);
const SourceResultsSchema = zod_1.z.object({
    path: zod_1.z.string(),
    type: BitbucketSourceTypeSchema,
    commit: zod_1.z.object({
        hash: zod_1.z.string(),
    }),
});
const PagedSchema = zod_1.z.object({
    page: zod_1.z.number().optional(),
    pagelen: zod_1.z.number(),
    size: zod_1.z.number().optional(),
    next: zod_1.z.string().optional(),
});
exports.PagedSourceResultsSchema = PagedSchema.extend({
    values: zod_1.z.array(SourceResultsSchema),
});
exports.RepoInfo = zod_1.z
    .object({
    parent: zod_1.z.unknown().optional().catch(undefined),
    mainbranch: zod_1.z.object({
        name: zod_1.z.string(),
    }),
    has_issues: zod_1.z.boolean().catch(() => {
        return false;
    }),
    uuid: zod_1.z.string(),
    full_name: zod_1.z
        .string()
        .regex(/^[^/]+\/[^/]+$/, 'Expected repository full_name to be in the format "owner/repo"'),
    is_private: zod_1.z.boolean().catch(() => {
        logger_1.logger.once.warn('Bitbucket: "is_private" field missing from repo info');
        return true;
    }),
    project: zod_1.z
        .object({
        name: zod_1.z.string(),
    })
        .nullable()
        .catch(null),
})
    .transform((repoInfoBody) => {
    const isFork = !!repoInfoBody.parent;
    const [owner, name] = repoInfoBody.full_name.split('/');
    return {
        isFork,
        owner,
        name,
        mainbranch: repoInfoBody.mainbranch.name,
        mergeMethod: 'merge',
        has_issues: repoInfoBody.has_issues,
        uuid: repoInfoBody.uuid,
        is_private: repoInfoBody.is_private,
        projectName: repoInfoBody.project?.name,
    };
});
exports.Repositories = zod_1.z
    .object({
    values: (0, schema_utils_1.LooseArray)(exports.RepoInfo),
})
    .transform((body) => body.values);
const TaskState = zod_1.z.union([zod_1.z.literal('RESOLVED'), zod_1.z.literal('UNRESOLVED')]);
const PrTask = zod_1.z.object({
    id: zod_1.z.number(),
    state: TaskState,
    content: zod_1.z.object({
        raw: zod_1.z.string(),
    }),
});
exports.UnresolvedPrTasks = zod_1.z
    .object({
    values: zod_1.z.array(PrTask),
})
    .transform((data) => data.values.filter((task) => task.state === 'UNRESOLVED'));
//# sourceMappingURL=schema.js.map