"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.REOPEN_PR_COMMENT_KEYWORD = void 0;
exports.ensureComment = ensureComment;
exports.reopenComments = reopenComments;
exports.ensureCommentRemoval = ensureCommentRemoval;
const logger_1 = require("../../../logger");
const bitbucket_1 = require("../../../util/http/bitbucket");
exports.REOPEN_PR_COMMENT_KEYWORD = 'reopen!';
const bitbucketHttp = new bitbucket_1.BitbucketHttp();
async function getComments(config, prNo) {
    const comments = (await bitbucketHttp.getJsonUnchecked(`/2.0/repositories/${config.repository}/pullrequests/${prNo}/comments`, {
        paginate: true,
    })).body.values;
    logger_1.logger.debug(`Found ${comments.length} comments`);
    return comments;
}
async function addComment(config, prNo, raw) {
    await bitbucketHttp.postJson(`/2.0/repositories/${config.repository}/pullrequests/${prNo}/comments`, {
        body: { content: { raw } },
    });
}
async function editComment(config, prNo, commentId, raw) {
    await bitbucketHttp.putJson(`/2.0/repositories/${config.repository}/pullrequests/${prNo}/comments/${commentId}`, {
        body: { content: { raw } },
    });
}
async function deleteComment(config, prNo, commentId) {
    await bitbucketHttp.deleteJson(`/2.0/repositories/${config.repository}/pullrequests/${prNo}/comments/${commentId}`);
}
async function ensureComment({ config, number: prNo, topic, content, }) {
    try {
        const comments = await getComments(config, prNo);
        let body;
        let commentId;
        let commentNeedsUpdating;
        if (topic) {
            logger_1.logger.debug(`Ensuring comment "${topic}" in #${prNo}`);
            body = `### ${topic}\n\n${content}`;
            comments.forEach((comment) => {
                if (comment.content.raw.startsWith(`### ${topic}\n\n`)) {
                    commentId = comment.id;
                    commentNeedsUpdating = comment.content.raw !== body;
                }
            });
        }
        else {
            logger_1.logger.debug(`Ensuring content-only comment in #${prNo}`);
            body = `${content}`;
            comments.forEach((comment) => {
                if (comment.content.raw === body) {
                    commentId = comment.id;
                    commentNeedsUpdating = false;
                }
            });
        }
        // sanitize any language that isn't supported by Bitbucket Cloud
        body = sanitizeCommentBody(body);
        if (!commentId) {
            await addComment(config, prNo, body);
            logger_1.logger.info({ repository: config.repository, prNo, topic }, 'Comment added');
        }
        else if (commentNeedsUpdating) {
            await editComment(config, prNo, commentId, body);
            logger_1.logger.debug({ repository: config.repository, prNo }, 'Comment updated');
        }
        else {
            logger_1.logger.debug('Comment is already update-to-date');
        }
        return true;
    }
    catch (err) /* v8 ignore start */ {
        logger_1.logger.warn({ err }, 'Error ensuring comment');
        return false;
    } /* v8 ignore stop */
}
async function reopenComments(config, prNo) {
    const comments = await getComments(config, prNo);
    const reopenComments = comments.filter((comment) => comment.content.raw.startsWith(exports.REOPEN_PR_COMMENT_KEYWORD));
    return reopenComments;
}
async function ensureCommentRemoval(config, deleteConfig) {
    try {
        const { number: prNo } = deleteConfig;
        const key = deleteConfig.type === 'by-topic'
            ? deleteConfig.topic
            : deleteConfig.content;
        logger_1.logger.debug(`Ensuring comment "${key}" in #${prNo} is removed`);
        const comments = await getComments(config, prNo);
        let commentId = undefined;
        if (deleteConfig.type === 'by-topic') {
            const byTopic = (comment) => comment.content.raw.startsWith(`### ${deleteConfig.topic}\n\n`);
            commentId = comments.find(byTopic)?.id;
        }
        else if (deleteConfig.type === 'by-content') {
            const byContent = (comment) => comment.content.raw.trim() === deleteConfig.content;
            commentId = comments.find(byContent)?.id;
        }
        if (commentId) {
            await deleteComment(config, prNo, commentId);
        }
    }
    catch (err) /* v8 ignore start */ {
        logger_1.logger.warn({ err }, 'Error ensuring comment removal');
    } /* v8 ignore stop */
}
function sanitizeCommentBody(body) {
    return body
        .replace('checking the rebase/retry box above', 'renaming this PR to start with "rebase!"')
        .replace('rename this PR to get a fresh replacement', 'add a comment starting with "reopen!" to get a fresh replacement');
}
//# sourceMappingURL=comments.js.map