"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BITBUCKET_INVALID_REVIEWERS_EXCEPTION = void 0;
exports.prInfo = prInfo;
exports.isInvalidReviewersResponse = isInvalidReviewersResponse;
exports.getInvalidReviewers = getInvalidReviewers;
exports.getRepoGitUrl = getRepoGitUrl;
exports.getExtraCloneOpts = getExtraCloneOpts;
const tslib_1 = require("tslib");
// SEE for the reference https://github.com/renovatebot/renovate/blob/c3e9e572b225085448d94aa121c7ec81c14d3955/lib/platform/bitbucket/utils.js
const node_url_1 = require("node:url");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const error_messages_1 = require("../../../constants/error-messages");
const logger_1 = require("../../../logger");
const git = tslib_1.__importStar(require("../../../util/git"));
const url_1 = require("../../../util/url");
const pr_body_1 = require("../pr-body");
exports.BITBUCKET_INVALID_REVIEWERS_EXCEPTION = 'com.atlassian.bitbucket.pull.InvalidPullRequestReviewersException';
// https://docs.atlassian.com/bitbucket-server/rest/6.0.0/bitbucket-rest.html#idp250
const prStateMapping = {
    MERGED: 'merged',
    DECLINED: 'closed',
    OPEN: 'open',
};
function prInfo(pr) {
    return {
        version: pr.version,
        number: pr.id,
        bodyStruct: (0, pr_body_1.getPrBodyStruct)(pr.description),
        sourceBranch: pr.fromRef.displayId,
        targetBranch: pr.toRef.displayId,
        title: pr.title,
        state: prStateMapping[pr.state],
        createdAt: pr.createdDate,
    };
}
function isInvalidReviewersResponse(err) {
    const errors = err?.response?.body?.errors ?? [];
    return (errors.length > 0 &&
        errors.every((error) => error.exceptionName === exports.BITBUCKET_INVALID_REVIEWERS_EXCEPTION));
}
function getInvalidReviewers(err) {
    const errors = err?.response?.body?.errors ?? [];
    let invalidReviewers = [];
    for (const error of errors) {
        if (error.exceptionName === exports.BITBUCKET_INVALID_REVIEWERS_EXCEPTION) {
            invalidReviewers = invalidReviewers.concat(error.reviewerErrors
                ?.map(({ context }) => context)
                .filter(is_1.default.nonEmptyString) ?? []);
        }
    }
    return invalidReviewers;
}
function generateUrlFromEndpoint(defaultEndpoint, opts, repository) {
    const url = new node_url_1.URL(defaultEndpoint);
    const generatedUrl = git.getUrl({
        protocol: url.protocol,
        // TODO: types (#22198)
        auth: `${opts.username}:${opts.password}`,
        host: `${url.host}${url.pathname}${
        /* v8 ignore start */
        url.pathname.endsWith('/') ? '' : '/'
        /* v8 ignore stop */
        }scm`,
        repository,
    });
    logger_1.logger.debug(`Using generated endpoint URL: ${generatedUrl}`);
    return generatedUrl;
}
function injectAuth(url, opts) {
    const repoUrl = (0, url_1.parseUrl)(url);
    if (!repoUrl) {
        logger_1.logger.debug(`Invalid url: ${url}`);
        throw new Error(error_messages_1.CONFIG_GIT_URL_UNAVAILABLE);
    }
    if (!opts.token && opts.username && opts.password) {
        repoUrl.username = opts.username;
        repoUrl.password = opts.password;
    }
    return repoUrl.toString();
}
function getRepoGitUrl(repository, defaultEndpoint, gitUrl, info, opts) {
    if (gitUrl === 'ssh') {
        const sshUrl = info.links.clone?.find(({ name }) => name === 'ssh');
        if (sshUrl === undefined) {
            throw new Error(error_messages_1.CONFIG_GIT_URL_UNAVAILABLE);
        }
        logger_1.logger.debug(`Using ssh URL: ${sshUrl.href}`);
        return sshUrl.href;
    }
    let cloneUrl = info.links.clone?.find(({ name }) => name === 'http');
    if (cloneUrl) {
        // Inject auth into the API provided URL
        return injectAuth(cloneUrl.href, opts);
    }
    // Http access might be disabled, try to find ssh url in this case
    cloneUrl = info.links.clone?.find(({ name }) => name === 'ssh');
    if (gitUrl === 'endpoint' || !cloneUrl) {
        return generateUrlFromEndpoint(defaultEndpoint, opts, repository);
    }
    // SSH urls can be used directly
    return cloneUrl.href;
}
function getExtraCloneOpts(opts) {
    if (opts.token) {
        return {
            '-c': `http.extraheader=Authorization: Bearer ${opts.token}`,
        };
    }
    return {};
}
//# sourceMappingURL=utils.js.map