"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BbsPrCache = void 0;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const dequal_1 = require("dequal");
const luxon_1 = require("luxon");
const logger_1 = require("../../../logger");
const memCache = tslib_1.__importStar(require("../../../util/cache/memory"));
const repository_1 = require("../../../util/cache/repository");
const url_1 = require("../../../util/url");
const utils_1 = require("./utils");
class BbsPrCache {
    projectKey;
    repo;
    ignorePrAuthor;
    author;
    cache;
    items = [];
    constructor(projectKey, repo, ignorePrAuthor, author) {
        this.projectKey = projectKey;
        this.repo = repo;
        this.ignorePrAuthor = ignorePrAuthor;
        this.author = author;
        const repoCache = (0, repository_1.getCache)();
        repoCache.platform ??= {};
        repoCache.platform.bitbucketServer ??= {};
        let pullRequestCache = repoCache.platform.bitbucketServer
            .pullRequestsCache;
        if (!pullRequestCache || pullRequestCache.author !== author) {
            pullRequestCache = {
                items: {},
                updatedDate: null,
                author,
            };
        }
        repoCache.platform.bitbucketServer.pullRequestsCache = pullRequestCache;
        this.cache = pullRequestCache;
        this.updateItems();
    }
    static async init(http, projectKey, repo, ignorePrAuthor, author) {
        const res = new BbsPrCache(projectKey, repo, ignorePrAuthor, author);
        const isSynced = memCache.get('bbs-pr-cache-synced');
        if (!isSynced) {
            await res.sync(http);
            memCache.set('bbs-pr-cache-synced', true);
        }
        return res;
    }
    getPrs() {
        return this.items;
    }
    static async getPrs(http, projectKey, repo, ignorePrAuthor, author) {
        const prCache = await BbsPrCache.init(http, projectKey, repo, ignorePrAuthor, author);
        return prCache.getPrs();
    }
    setPr(item) {
        this.cache.items[item.number] = item;
        this.updateItems();
    }
    static async setPr(http, projectKey, repo, ignorePrAuthor, author, item) {
        const prCache = await BbsPrCache.init(http, projectKey, repo, ignorePrAuthor, author);
        prCache.setPr(item);
    }
    reconcile(rawItems) {
        logger_1.logger.debug('reconciled');
        const { items } = this.cache;
        let { updatedDate } = this.cache;
        const cacheTime = updatedDate ? luxon_1.DateTime.fromMillis(updatedDate) : null;
        let needNextPage = true;
        for (const rawItem of rawItems) {
            const id = rawItem.id;
            const newItem = (0, utils_1.prInfo)(rawItem);
            const oldItem = items[id];
            if ((0, dequal_1.dequal)(oldItem, newItem)) {
                needNextPage = false;
                continue;
            }
            items[id] = newItem;
            const itemTime = luxon_1.DateTime.fromMillis(rawItem.updatedDate);
            if (!cacheTime || itemTime > cacheTime) {
                updatedDate = rawItem.updatedDate;
            }
        }
        this.cache.updatedDate = updatedDate;
        return needNextPage;
    }
    async sync(http) {
        const searchParams = {
            state: 'ALL',
            limit: this.items.length ? '20' : '100',
        };
        if (!this.ignorePrAuthor && is_1.default.string(this.author)) {
            searchParams['role.1'] = 'AUTHOR';
            searchParams['username.1'] = this.author;
        }
        let query = (0, url_1.getQueryString)(searchParams);
        while (query) {
            const res = await http.getJsonUnchecked(`./rest/api/1.0/projects/${this.projectKey}/repos/${this.repo}/pull-requests?${query}`, {
                memCache: false,
            });
            const needNextPage = this.reconcile(res.body.values);
            if (!needNextPage) {
                break;
            }
            if (res.body.nextPageStart) {
                searchParams.start = res.body.nextPageStart.toString();
            }
            else {
                query = null;
            }
        }
        this.updateItems();
        return this;
    }
    /**
     * Ensure the pr cache starts with the most recent PRs.
     * JavaScript ensures that the cache is sorted by PR number.
     */
    updateItems() {
        this.items = Object.values(this.cache.items).reverse();
    }
}
exports.BbsPrCache = BbsPrCache;
//# sourceMappingURL=pr-cache.js.map