"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.extractHelmChart = extractHelmChart;
exports.extractGitSource = extractGitSource;
exports.extractGithubReleaseSource = extractGithubReleaseSource;
exports.parseVendir = parseVendir;
exports.extractPackageFile = extractPackageFile;
const logger_1 = require("../../../logger");
const url_1 = require("../../../util/git/url");
const yaml_1 = require("../../../util/yaml");
const git_refs_1 = require("../../datasource/git-refs");
const github_releases_1 = require("../../datasource/github-releases");
const helm_1 = require("../../datasource/helm");
const extract_1 = require("../dockerfile/extract");
const oci_1 = require("../helmv3/oci");
const schema_1 = require("./schema");
function extractHelmChart(helmChart, aliases) {
    if ((0, oci_1.isOCIRegistry)(helmChart.repository.url)) {
        const dep = (0, extract_1.getDep)(`${(0, oci_1.removeOCIPrefix)(helmChart.repository.url)}/${helmChart.name}:${helmChart.version}`, false, aliases);
        return {
            ...dep,
            depName: helmChart.name,
            depType: 'HelmChart',
            // https://github.com/helm/helm/issues/10312
            // https://github.com/helm/helm/issues/10678
            pinDigests: false,
        };
    }
    return {
        depName: helmChart.name,
        currentValue: helmChart.version,
        depType: 'HelmChart',
        registryUrls: [helmChart.repository.url],
        datasource: helm_1.HelmDatasource.id,
    };
}
function extractGitSource(gitSource) {
    const httpUrl = (0, url_1.getHttpUrl)(gitSource.url);
    return {
        depName: httpUrl,
        packageName: httpUrl,
        depType: 'GitSource',
        currentValue: gitSource.ref,
        registryUrls: [httpUrl],
        datasource: git_refs_1.GitRefsDatasource.id,
    };
}
function extractGithubReleaseSource(githubRelease) {
    return {
        depName: githubRelease.slug,
        packageName: githubRelease.slug,
        depType: 'GithubRelease',
        currentValue: githubRelease.tag,
        datasource: github_releases_1.GithubReleasesDatasource.id,
    };
}
function parseVendir(content, packageFile) {
    try {
        return (0, yaml_1.parseSingleYaml)(content, {
            customSchema: schema_1.Vendir,
            removeTemplates: true,
        });
    }
    catch {
        logger_1.logger.debug({ packageFile }, 'Error parsing vendir.yml file');
        return null;
    }
}
function extractPackageFile(content, packageFile, config) {
    logger_1.logger.trace(`vendir.extractPackageFile(${packageFile})`);
    const deps = [];
    const pkg = parseVendir(content, packageFile);
    if (!pkg) {
        return null;
    }
    // grab the helm charts
    const contents = pkg.directories.flatMap((directory) => directory.contents);
    for (const content of contents) {
        if ('helmChart' in content && content.helmChart) {
            const dep = extractHelmChart(content.helmChart, config.registryAliases);
            if (dep) {
                deps.push(dep);
            }
        }
        else if ('git' in content && content.git) {
            const dep = extractGitSource(content.git);
            if (dep) {
                deps.push(dep);
            }
        }
        else if ('githubRelease' in content && content.githubRelease) {
            const dep = extractGithubReleaseSource(content.githubRelease);
            if (dep) {
                deps.push(dep);
            }
        }
    }
    if (!deps.length) {
        return null;
    }
    return { deps };
}
//# sourceMappingURL=extract.js.map