"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HelmReleaseExtractor = void 0;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const logger_1 = require("../../../../../logger");
const url_1 = require("../../../../../util/url");
const helm_1 = require("../../../../datasource/helm");
const extract_1 = require("../../../dockerfile/extract");
const oci_1 = require("../../../helmv3/oci");
const base_1 = require("../../base");
const util_1 = require("../../util");
class HelmReleaseExtractor extends base_1.DependencyExtractor {
    getCheckList() {
        return [`"helm_release"`];
    }
    extract(hclMap, _locks, config) {
        const dependencies = [];
        const helmReleases = hclMap?.resource?.helm_release;
        if (is_1.default.nullOrUndefined(helmReleases)) {
            return [];
        }
        // istanbul ignore if
        if (!is_1.default.plainObject(helmReleases)) {
            logger_1.logger.debug({ helmReleases }, 'Terraform: unexpected `helmReleases` value');
            return [];
        }
        for (const helmRelease of Object.values(helmReleases).flat()) {
            const dep = {
                currentValue: helmRelease.version,
                depType: 'helm_release',
                depName: helmRelease.chart,
                datasource: helm_1.HelmDatasource.id,
            };
            dependencies.push(dep);
            if (!is_1.default.nonEmptyString(helmRelease.chart)) {
                dep.skipReason = 'invalid-name';
            }
            else if ((0, oci_1.isOCIRegistry)(helmRelease.chart)) {
                // For oci charts, we remove the oci:// and use the docker datasource
                dep.depName = (0, oci_1.removeOCIPrefix)(helmRelease.chart);
                this.processOCI(dep.depName, config, dep);
            }
            else if ((0, util_1.checkIfStringIsPath)(helmRelease.chart)) {
                dep.skipReason = 'local-chart';
            }
            else if (is_1.default.nonEmptyString(helmRelease.repository)) {
                if ((0, oci_1.isOCIRegistry)(helmRelease.repository)) {
                    // For oci charts, we remove the oci:// and use the docker datasource
                    this.processOCI((0, url_1.joinUrlParts)((0, oci_1.removeOCIPrefix)(helmRelease.repository), helmRelease.chart), config, dep);
                }
                else {
                    dep.registryUrls = [helmRelease.repository];
                }
            }
        }
        return dependencies;
    }
    processOCI(depName, config, dep) {
        const { packageName, datasource } = (0, extract_1.getDep)(depName, false, config.registryAliases);
        dep.packageName = packageName;
        dep.datasource = datasource;
    }
}
exports.HelmReleaseExtractor = HelmReleaseExtractor;
//# sourceMappingURL=helm-release.js.map