"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.extractPackageFile = extractPackageFile;
const logger_1 = require("../../../logger");
const url_1 = require("../../../util/url");
const git_tags_1 = require("../../datasource/git-tags");
const github_tags_1 = require("../../datasource/github-tags");
const puppet_forge_1 = require("../../datasource/puppet-forge");
const common_1 = require("./common");
const puppetfile_parser_1 = require("./puppetfile-parser");
function parseForgeDependency(module, forgeUrl) {
    const dep = {
        depName: module.name,
        datasource: puppet_forge_1.PuppetForgeDatasource.id,
        packageName: module.name,
        currentValue: module.version,
    };
    if (forgeUrl) {
        dep.registryUrls = [forgeUrl];
    }
    return dep;
}
function parseGitDependency(module) {
    const moduleName = module.name;
    const git = module.tags?.get('git');
    const tag = module.tags?.get('tag');
    if (!git || !tag) {
        return {
            depName: moduleName,
            sourceUrl: git,
            skipReason: 'invalid-version',
        };
    }
    const parsedUrl = (0, url_1.parseUrl)(git);
    const githubUrl = (0, common_1.isGithubUrl)(git, parsedUrl);
    if (githubUrl && parsedUrl && parsedUrl.protocol !== 'https:') {
        logger_1.logger.debug(`Access to github is only allowed for https, your url was: ${git}`);
        return {
            depName: moduleName,
            sourceUrl: git,
            skipReason: 'invalid-url',
        };
    }
    const gitOwnerRepo = (0, common_1.parseGitOwnerRepo)(git, githubUrl);
    if (!gitOwnerRepo) {
        // failed to parse git url
        return {
            depName: moduleName,
            sourceUrl: git,
            skipReason: 'invalid-url',
        };
    }
    const packageDependency = {
        depName: moduleName,
        packageName: git,
        sourceUrl: git,
        gitRef: true,
        currentValue: tag,
        datasource: git_tags_1.GitTagsDatasource.id,
    };
    if (githubUrl) {
        packageDependency.packageName = gitOwnerRepo;
        packageDependency.datasource = github_tags_1.GithubTagsDatasource.id;
    }
    return packageDependency;
}
function isGitModule(module) {
    return module.tags?.has('git') ?? false;
}
function extractPackageFile(content) {
    logger_1.logger.trace('puppet.extractPackageFile()');
    const puppetFile = (0, puppetfile_parser_1.parsePuppetfile)(content);
    const deps = [];
    for (const forgeUrl of puppetFile.getForges()) {
        for (const module of puppetFile.getModulesOfForge(forgeUrl)) {
            let packageDependency;
            if (isGitModule(module)) {
                packageDependency = parseGitDependency(module);
            }
            else {
                packageDependency = parseForgeDependency(module, forgeUrl);
            }
            if (module.skipReason) {
                // the PuppetfileModule skip reason is dominant over the packageDependency skip reason
                packageDependency.skipReason = module.skipReason;
            }
            deps.push(packageDependency);
        }
    }
    return deps.length ? { deps } : null;
}
//# sourceMappingURL=extract.js.map