"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.depTypes = void 0;
exports.parsePEP508 = parsePEP508;
exports.pep508ToPackageDependency = pep508ToPackageDependency;
exports.parseDependencyGroupRecord = parseDependencyGroupRecord;
exports.parseDependencyList = parseDependencyList;
exports.parsePyProject = parsePyProject;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const logger_1 = require("../../../logger");
const regex_1 = require("../../../util/regex");
const toml_1 = require("../../../util/toml");
const pypi_1 = require("../../datasource/pypi");
const common_1 = require("../../datasource/pypi/common");
const schema_1 = require("./schema");
const pep508Regex = (0, regex_1.regEx)(/^(?<packageName>[A-Z0-9._-]+)\s*(\[(?<extras>[A-Z0-9\s,._-]+)\])?\s*(?<currentValue>[^;]+)?(;\s*(?<marker>.*))?/i);
exports.depTypes = {
    dependencies: 'project.dependencies',
    optionalDependencies: 'project.optional-dependencies',
    dependencyGroups: 'dependency-groups',
    pdmDevDependencies: 'tool.pdm.dev-dependencies',
    uvDevDependencies: 'tool.uv.dev-dependencies',
    uvSources: 'tool.uv.sources',
    buildSystemRequires: 'build-system.requires',
};
function parsePEP508(value) {
    if (is_1.default.nullOrUndefined(value)) {
        return null;
    }
    const regExpExec = pep508Regex.exec(value);
    if (is_1.default.nullOrUndefined(regExpExec) ||
        is_1.default.nullOrUndefined(regExpExec?.groups)) {
        logger_1.logger.trace(`Pep508 could not be extracted`);
        return null;
    }
    const result = {
        packageName: regExpExec.groups.packageName,
    };
    if (is_1.default.nonEmptyString(regExpExec.groups.currentValue)) {
        if (regExpExec.groups.currentValue.startsWith('(') &&
            regExpExec.groups.currentValue.endsWith(')')) {
            result.currentValue = regExpExec.groups.currentValue.slice(1, -1).trim();
        }
        else {
            result.currentValue = regExpExec.groups.currentValue;
        }
    }
    if (is_1.default.nonEmptyString(regExpExec.groups.marker)) {
        result.marker = regExpExec.groups.marker;
    }
    if (is_1.default.nonEmptyString(regExpExec.groups.extras)) {
        // trim to remove allowed whitespace between brackets
        result.extras = regExpExec.groups.extras.split(',').map((e) => e.trim());
    }
    return result;
}
function pep508ToPackageDependency(depType, value) {
    const parsed = parsePEP508(value);
    if (is_1.default.nullOrUndefined(parsed)) {
        return null;
    }
    const dep = {
        packageName: (0, common_1.normalizePythonDepName)(parsed.packageName),
        depName: parsed.packageName,
        datasource: pypi_1.PypiDatasource.id,
        depType,
    };
    if (is_1.default.nullOrUndefined(parsed.currentValue)) {
        dep.skipReason = 'unspecified-version';
    }
    else {
        dep.currentValue = parsed.currentValue;
        if (parsed.currentValue.startsWith('==')) {
            dep.currentVersion = parsed.currentValue.replace((0, regex_1.regEx)(/^==\s*/), '');
        }
    }
    return dep;
}
function parseDependencyGroupRecord(depType, records) {
    if (is_1.default.nullOrUndefined(records)) {
        return [];
    }
    const deps = [];
    for (const [depGroup, pep508Strings] of Object.entries(records)) {
        for (const dep of parseDependencyList(depType, pep508Strings)) {
            deps.push({
                ...dep,
                depName: dep.packageName,
                managerData: { depGroup },
            });
        }
    }
    return deps;
}
function parseDependencyList(depType, list) {
    if (is_1.default.nullOrUndefined(list)) {
        return [];
    }
    const deps = [];
    for (const element of list) {
        const dep = pep508ToPackageDependency(depType, element);
        if (is_1.default.truthy(dep)) {
            deps.push(dep);
        }
    }
    return deps;
}
function parsePyProject(packageFile, content) {
    try {
        const jsonMap = (0, toml_1.parse)((0, toml_1.massage)(content));
        return schema_1.PyProjectSchema.parse(jsonMap);
    }
    catch (err) {
        logger_1.logger.debug({ packageFile, err }, `Failed to parse and validate pyproject file`);
        return null;
    }
}
//# sourceMappingURL=utils.js.map