"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.extractPackageFile = extractPackageFile;
exports.processModule = processModule;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const logger_1 = require("../../../logger");
const regex_1 = require("../../../util/regex");
const yaml_1 = require("../../../util/yaml");
const go_1 = require("../../datasource/go");
const schema_1 = require("./schema");
function extractPackageFile(content, packageFile, _config) {
    let definition = null;
    try {
        const yaml = (0, yaml_1.parseSingleYaml)(content);
        const parsed = schema_1.OCBConfigSchema.safeParse(yaml);
        if (!parsed.success) {
            logger_1.logger.trace({ packageFile, error: parsed.error }, 'Failed to parse OCB schema');
            return null;
        }
        definition = parsed.data;
    }
    catch (error) {
        logger_1.logger.debug({ packageFile, error }, 'OCB manager failed to parse file as YAML');
        return null;
    }
    const deps = [];
    if (definition.dist.otelcol_version) {
        deps.push({
            datasource: go_1.GoDatasource.id,
            depType: 'collector',
            depName: 'go.opentelemetry.io/collector',
            currentValue: definition.dist.otelcol_version,
            extractVersion: '^v(?<version>\\S+)',
        });
    }
    deps.push(...processModule(definition.connectors, 'connectors'));
    deps.push(...processModule(definition.exporters, 'exports'));
    deps.push(...processModule(definition.extensions, 'extensions'));
    deps.push(...processModule(definition.processors, 'processors'));
    deps.push(...processModule(definition.providers, 'providers'));
    deps.push(...processModule(definition.receivers, 'receivers'));
    return {
        packageFileVersion: definition.dist.version,
        deps,
    };
}
function processModule(module, depType) {
    const deps = [];
    if (is_1.default.nullOrUndefined(module)) {
        return deps;
    }
    for (const element of module) {
        const [depName, currentValue] = element.gomod.trim().split((0, regex_1.regEx)(/\s+/));
        deps.push({
            datasource: go_1.GoDatasource.id,
            depType,
            depName,
            currentValue,
        });
    }
    return deps;
}
//# sourceMappingURL=extract.js.map