"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.extractPackageFile = extractPackageFile;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const xmldoc_1 = require("xmldoc");
const logger_1 = require("../../../logger");
const fs_1 = require("../../../util/fs");
const nuget_1 = require("../../datasource/nuget");
const extract_1 = require("../dockerfile/extract");
const global_manifest_1 = require("./extract/global-manifest");
const util_1 = require("./util");
/**
 * https://docs.microsoft.com/en-us/nuget/concepts/package-versioning
 * This article mentions that  Nuget 3.x and later tries to restore the lowest possible version
 * regarding to given version range.
 * 1.3.4 equals [1.3.4,)
 */
const elemNames = new Set([
    'PackageReference',
    'PackageVersion',
    'DotNetCliToolReference',
    'GlobalPackageReference',
]);
function extractDepsFromXml(xmlNode) {
    const results = [];
    const vars = new Map();
    const todo = [xmlNode];
    while (todo.length) {
        const child = todo.pop();
        const { name, attr } = child;
        if (name === 'ContainerBaseImage') {
            const { depName, ...dep } = (0, extract_1.getDep)(child.val, true);
            if (is_1.default.nonEmptyStringAndNotWhitespace(depName)) {
                results.push({ ...dep, depName, depType: 'docker' });
            }
        }
        else if (elemNames.has(name)) {
            const depName = attr?.Include || attr?.Update;
            if (!depName) {
                continue;
            }
            const dep = {
                datasource: nuget_1.NugetDatasource.id,
                depType: 'nuget',
                depName,
            };
            let currentValue = attr?.Version ??
                attr?.version ??
                child.valueWithPath('Version') ??
                attr?.VersionOverride ??
                child.valueWithPath('VersionOverride');
            if (!is_1.default.nonEmptyStringAndNotWhitespace(currentValue)) {
                dep.skipReason = 'invalid-version';
            }
            let sharedVariableName;
            currentValue = currentValue
                ?.trim()
                ?.replace(/^\$\((\w+)\)$/, (match, key) => {
                sharedVariableName = key;
                const val = vars.get(key);
                if (val) {
                    return val;
                }
                return match;
            });
            if (sharedVariableName) {
                if (currentValue === `$(${sharedVariableName})`) {
                    // this means that be failed to find/replace the variable
                    dep.skipReason = 'contains-variable';
                }
                else {
                    dep.sharedVariableName = sharedVariableName;
                }
            }
            dep.currentValue = currentValue;
            results.push(dep);
        }
        else if (name === 'Sdk') {
            const depName = attr?.Name;
            const version = attr?.Version;
            // if sdk element is present it will always have the Name field but the Version is an optional field
            if (depName && version) {
                results.push({
                    depName,
                    currentValue: version,
                    depType: 'msbuild-sdk',
                    datasource: nuget_1.NugetDatasource.id,
                });
            }
        }
        else if (name === 'Import') {
            const depName = attr?.Sdk;
            const version = attr?.Version;
            if (depName && version) {
                results.push({
                    depName,
                    currentValue: version,
                    depType: 'msbuild-sdk',
                    datasource: nuget_1.NugetDatasource.id,
                });
            }
        }
        else {
            if (name === 'Project') {
                if (attr?.Sdk) {
                    const str = attr?.Sdk;
                    const [name, version] = str.split('/');
                    if (name && version) {
                        results.push({
                            depName: name,
                            depType: 'msbuild-sdk',
                            currentValue: version,
                            datasource: nuget_1.NugetDatasource.id,
                        });
                    }
                }
                const propertyGroup = child.childNamed('PropertyGroup');
                if (propertyGroup) {
                    for (const propChild of propertyGroup.children) {
                        if ((0, util_1.isXmlElement)(propChild)) {
                            const { name, val } = propChild;
                            if (!['Version', 'TargetFramework'].includes(name)) {
                                vars.set(name, val);
                            }
                        }
                    }
                }
            }
            todo.push(...child.children.filter(util_1.isXmlElement));
        }
    }
    return results;
}
async function extractPackageFile(content, packageFile, _config) {
    logger_1.logger.trace(`nuget.extractPackageFile(${packageFile})`);
    const registries = await (0, util_1.getConfiguredRegistries)(packageFile);
    if (packageFile.endsWith('dotnet-tools.json')) {
        const deps = [];
        let manifest;
        try {
            manifest = JSON.parse(content);
        }
        catch {
            logger_1.logger.debug({ packageFile }, `Invalid JSON`);
            return null;
        }
        if (manifest.version !== 1) {
            logger_1.logger.debug({ packageFile }, 'Unsupported dotnet tools version');
            return null;
        }
        for (const depName of Object.keys(manifest.tools ?? {})) {
            const tool = manifest.tools[depName];
            const currentValue = tool.version;
            const dep = {
                depType: 'nuget',
                depName,
                currentValue,
                datasource: nuget_1.NugetDatasource.id,
            };
            (0, util_1.applyRegistries)(dep, registries);
            deps.push(dep);
        }
        return deps.length ? { deps } : null;
    }
    if (packageFile.endsWith('global.json')) {
        return (0, global_manifest_1.extractMsbuildGlobalManifest)(content, packageFile, registries);
    }
    let deps = [];
    let packageFileVersion;
    try {
        const parsedXml = new xmldoc_1.XmlDocument(content);
        deps = extractDepsFromXml(parsedXml).map((dep) => (0, util_1.applyRegistries)(dep, registries));
        packageFileVersion = (0, util_1.findVersion)(parsedXml)?.val;
    }
    catch (err) {
        logger_1.logger.debug({ err, packageFile }, `Failed to parse XML`);
    }
    if (!deps.length) {
        return null;
    }
    const res = { deps, packageFileVersion };
    const lockFileName = (0, fs_1.getSiblingFileName)(packageFile, 'packages.lock.json');
    // istanbul ignore if
    if (await (0, fs_1.localPathExists)(lockFileName)) {
        res.lockFiles = [lockFileName];
    }
    return res;
}
//# sourceMappingURL=extract.js.map