"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.extractPackageJson = extractPackageJson;
const tslib_1 = require("tslib");
const parse_overrides_1 = require("@pnpm/parse-overrides");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const error_messages_1 = require("../../../../../constants/error-messages");
const logger_1 = require("../../../../../logger");
const regex_1 = require("../../../../../util/regex");
const dependency_1 = require("./dependency");
const node_1 = require("./node");
const overrides_1 = require("./overrides");
function extractPackageJson(packageJson, packageFile) {
    logger_1.logger.trace(`npm.extractPackageJson(${packageFile})`);
    const deps = [];
    if (packageJson._id && packageJson._args && packageJson._from) {
        logger_1.logger.debug({ packageFile }, 'Ignoring vendorised package.json');
        return null;
    }
    if (packageFile !== 'package.json' && packageJson.renovate) {
        const error = new Error(error_messages_1.CONFIG_VALIDATION);
        error.validationSource = packageFile;
        error.validationError =
            'Nested package.json must not contain Renovate configuration. Please use `packageRules` with `matchFileNames` in your main config instead.';
        throw error;
    }
    const packageJsonName = packageJson.name;
    logger_1.logger.debug(`npm file ${packageFile} has name ${JSON.stringify(packageJsonName)}`);
    const packageFileVersion = packageJson.version;
    const depTypes = {
        dependencies: 'dependency',
        devDependencies: 'devDependency',
        optionalDependencies: 'optionalDependency',
        peerDependencies: 'peerDependency',
        engines: 'engine',
        volta: 'volta',
        resolutions: 'resolutions',
        packageManager: 'packageManager',
        overrides: 'overrides',
        pnpm: 'pnpm',
    };
    for (const depType of Object.keys(depTypes)) {
        let dependencies = packageJson[depType];
        if (dependencies) {
            try {
                if (depType === 'packageManager') {
                    const match = (0, regex_1.regEx)('^(?<name>.+)@(?<range>.+)$').exec(dependencies);
                    // istanbul ignore next
                    if (!match?.groups) {
                        break;
                    }
                    dependencies = { [match.groups.name]: match.groups.range };
                }
                for (const [key, val] of Object.entries(dependencies)) {
                    const depName = (0, dependency_1.parseDepName)(depType, key);
                    let dep = {
                        depType,
                        depName,
                    };
                    if (depName !== key) {
                        dep.managerData = { key };
                    }
                    if (depType === 'overrides' && !is_1.default.string(val)) {
                        // TODO: fix type #22198
                        deps.push(...(0, overrides_1.extractOverrideDepsRec)([depName], val));
                    }
                    else if (depType === 'pnpm' && depName === 'overrides') {
                        // pnpm overrides
                        // https://pnpm.io/package_json#pnpmoverrides
                        for (const [overridesKey, overridesVal] of Object.entries(val)) {
                            if (is_1.default.string(overridesVal)) {
                                // Newer flat syntax: `parent>parent>child`
                                const packageName = (0, parse_overrides_1.parsePkgAndParentSelector)(overridesKey).targetPkg.name;
                                dep = {
                                    depName: overridesKey,
                                    packageName,
                                    depType: 'pnpm.overrides',
                                    ...(0, dependency_1.extractDependency)(depName, packageName, overridesVal),
                                };
                                (0, node_1.setNodeCommitTopic)(dep);
                                // TODO: Is this expected? It's always 'overrides'.
                                dep.prettyDepType = depTypes[depName];
                                deps.push(dep);
                            }
                            else if (is_1.default.object(overridesVal)) {
                                // Older nested object syntax: `parent: { parent: { child: version } }`
                                deps.push(...(0, overrides_1.extractOverrideDepsRec)([overridesKey], overridesVal));
                            }
                        }
                    }
                    else {
                        // TODO: fix type #22198
                        dep = { ...dep, ...(0, dependency_1.extractDependency)(depType, depName, val) };
                        (0, node_1.setNodeCommitTopic)(dep);
                        dep.prettyDepType = depTypes[depType];
                        deps.push(dep);
                    }
                }
            }
            catch (err) /* istanbul ignore next */ {
                logger_1.logger.debug({ fileName: packageFile, depType, err }, 'Error parsing package.json');
                return null;
            }
        }
    }
    const extractedConstraints = (0, dependency_1.getExtractedConstraints)(deps);
    return {
        deps,
        extractedConstraints,
        packageFileVersion,
        managerData: {
            packageJsonName,
            hasPackageManager: is_1.default.nonEmptyStringAndNotWhitespace(packageJson.packageManager) ||
                is_1.default.nonEmptyObject(packageJson.devEngines?.packageManager),
            workspaces: packageJson.workspaces,
        },
    };
}
//# sourceMappingURL=package-file.js.map