"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.miseTooling = exports.asdfTooling = void 0;
const tslib_1 = require("tslib");
const regex_1 = require("../../../util/regex");
const github_releases_1 = require("../../datasource/github-releases");
const github_tags_1 = require("../../datasource/github-tags");
const hexpm_bob_1 = require("../../datasource/hexpm-bob");
const java_version_1 = require("../../datasource/java-version");
const node_version_1 = require("../../datasource/node-version");
const ruby_version_1 = require("../../datasource/ruby-version");
const regexVersioning = tslib_1.__importStar(require("../../versioning/regex"));
const semverVersioning = tslib_1.__importStar(require("../../versioning/semver"));
const upgradeable_tooling_1 = require("../asdf/upgradeable-tooling");
exports.asdfTooling = upgradeable_tooling_1.upgradeableTooling;
const miseCoreTooling = {
    bun: {
        misePluginUrl: 'https://mise.jdx.dev/lang/bun.html',
        config: {
            packageName: 'oven-sh/bun',
            datasource: github_releases_1.GithubReleasesDatasource.id,
            extractVersion: '^bun-v(?<version>\\S+)',
        },
    },
    deno: {
        misePluginUrl: 'https://mise.jdx.dev/lang/deno.html',
        config: {
            packageName: 'denoland/deno',
            datasource: github_releases_1.GithubReleasesDatasource.id,
            extractVersion: '^v(?<version>\\S+)',
        },
    },
    elixir: {
        misePluginUrl: 'https://mise.jdx.dev/lang/elixir.html',
        config: {
            datasource: hexpm_bob_1.HexpmBobDatasource.id,
        },
    },
    erlang: {
        misePluginUrl: 'https://mise.jdx.dev/lang/erlang.html',
        config: {
            packageName: 'erlang/otp',
            datasource: github_tags_1.GithubTagsDatasource.id,
            extractVersion: '^OTP-(?<version>\\S+)',
            versioning: `${regexVersioning.id}:^(?<major>\\d+?)\\.(?<minor>\\d+?)(\\.(?<patch>\\d+))?$`,
        },
    },
    go: {
        misePluginUrl: 'https://mise.jdx.dev/lang/go.html',
        config: {
            packageName: 'golang/go',
            datasource: github_tags_1.GithubTagsDatasource.id,
            extractVersion: '^go(?<version>\\S+)',
        },
    },
    java: {
        misePluginUrl: 'https://mise.jdx.dev/lang/java.html',
        config: (version) => {
            // no prefix is shorthand for openjdk
            const versionMatch = (0, regex_1.regEx)(/^(\d\S+)/).exec(version)?.[1];
            if (versionMatch) {
                return {
                    datasource: java_version_1.JavaVersionDatasource.id,
                    packageName: 'java-jdk',
                    currentValue: versionMatch,
                };
            }
            const openJdkMatches = (0, regex_1.regEx)(/^openjdk-(?<version>\d\S+)/).exec(version)?.groups;
            if (openJdkMatches) {
                return {
                    datasource: java_version_1.JavaVersionDatasource.id,
                    packageName: 'java-jdk',
                    currentValue: openJdkMatches.version,
                };
            }
            const adoptOpenJdkMatches = (0, regex_1.regEx)(/^adoptopenjdk-(?<version>\d\S+)/).exec(version)?.groups;
            if (adoptOpenJdkMatches) {
                return {
                    datasource: java_version_1.JavaVersionDatasource.id,
                    packageName: 'java-jdk',
                    currentValue: adoptOpenJdkMatches.version,
                };
            }
            const temurinJdkMatches = (0, regex_1.regEx)(/^temurin-(?<version>\d\S+)/).exec(version)?.groups;
            if (temurinJdkMatches) {
                return {
                    datasource: java_version_1.JavaVersionDatasource.id,
                    packageName: 'java-jdk',
                    currentValue: temurinJdkMatches.version,
                };
            }
            const correttoJdkMatches = (0, regex_1.regEx)(/^corretto-(?<version>\d\S+)/).exec(version)?.groups;
            if (correttoJdkMatches) {
                return {
                    datasource: java_version_1.JavaVersionDatasource.id,
                    packageName: 'java-jdk',
                    currentValue: correttoJdkMatches.version,
                };
            }
            const zuluJdkMatches = (0, regex_1.regEx)(/^zulu-(?<version>\d\S+)/).exec(version)?.groups;
            if (zuluJdkMatches) {
                return {
                    datasource: java_version_1.JavaVersionDatasource.id,
                    packageName: 'java-jdk',
                    currentValue: zuluJdkMatches.version,
                };
            }
            const oracleGraalvmJdkMatches = (0, regex_1.regEx)(/^oracle-graalvm-(?<version>\d\S+)/).exec(version)?.groups;
            if (oracleGraalvmJdkMatches) {
                return {
                    datasource: java_version_1.JavaVersionDatasource.id,
                    packageName: 'java-jdk',
                    currentValue: oracleGraalvmJdkMatches.version,
                };
            }
            return undefined;
        },
    },
    node: {
        misePluginUrl: 'https://mise.jdx.dev/lang/node.html',
        config: {
            packageName: 'nodejs',
            datasource: node_version_1.NodeVersionDatasource.id,
        },
    },
    python: {
        misePluginUrl: 'https://mise.jdx.dev/lang/python.html',
        config: {
            packageName: 'python/cpython',
            datasource: github_tags_1.GithubTagsDatasource.id,
            extractVersion: '^v(?<version>\\S+)',
        },
    },
    ruby: {
        misePluginUrl: 'https://mise.jdx.dev/lang/ruby.html',
        config: {
            packageName: 'ruby-version',
            datasource: ruby_version_1.RubyVersionDatasource.id,
            versioning: semverVersioning.id,
        },
    },
    rust: {
        misePluginUrl: 'https://mise.jdx.dev/lang/rust.html',
        config: {
            packageName: 'rust-lang/rust',
            datasource: github_tags_1.GithubTagsDatasource.id,
        },
    },
    swift: {
        misePluginUrl: 'https://mise.jdx.dev/lang/swift.html',
        config: {
            packageName: 'swift-lang/swift',
            datasource: github_releases_1.GithubReleasesDatasource.id,
            extractVersion: '^swift-(?<version>\\S+)',
        },
    },
    zig: {
        misePluginUrl: 'https://mise.jdx.dev/lang/zig.html',
        config: {
            packageName: 'ziglang/zig',
            datasource: github_tags_1.GithubTagsDatasource.id,
        },
    },
};
const miseRegistryTooling = {
    actionlint: {
        misePluginUrl: 'https://mise.jdx.dev/registry.html#tools',
        config: {
            packageName: 'rhysd/actionlint',
            datasource: github_releases_1.GithubReleasesDatasource.id,
            extractVersion: '^v(?<version>\\S+)',
        },
    },
    astro: {
        misePluginUrl: 'https://mise.jdx.dev/registry.html#tools',
        config: {
            packageName: 'astronomer/astro-cli',
            datasource: github_releases_1.GithubReleasesDatasource.id,
            extractVersion: '^v(?<version>\\S+)',
        },
    },
    'aws-cli': {
        misePluginUrl: 'https://mise.jdx.dev/registry.html#tools',
        config: {
            datasource: github_tags_1.GithubTagsDatasource.id,
            packageName: 'aws/aws-cli',
        },
    },
    'aws-vault': {
        misePluginUrl: 'https://mise.jdx.dev/registry.html#tools',
        config: {
            datasource: github_releases_1.GithubReleasesDatasource.id,
            packageName: '99designs/aws-vault',
            extractVersion: '^v(?<version>\\S+)',
        },
    },
    buf: {
        misePluginUrl: 'https://mise.jdx.dev/registry.html#tools',
        config: {
            packageName: 'bufbuild/buf',
            datasource: github_releases_1.GithubReleasesDatasource.id,
            extractVersion: '^v(?<version>\\S+)',
        },
    },
    ccache: {
        misePluginUrl: 'https://mise.jdx.dev/registry.html#tools',
        config: {
            packageName: 'ccache/ccache',
            datasource: github_releases_1.GithubReleasesDatasource.id,
            extractVersion: '^v(?<version>\\S+)',
        },
    },
    committed: {
        misePluginUrl: 'https://mise.jdx.dev/registry.html#tools',
        config: {
            packageName: 'crate-ci/committed',
            datasource: github_releases_1.GithubReleasesDatasource.id,
            extractVersion: '^v(?<version>\\S+)',
        },
    },
    consul: {
        misePluginUrl: 'https://mise.jdx.dev/registry.html#tools',
        config: {
            packageName: 'hashicorp/consul',
            datasource: github_releases_1.GithubReleasesDatasource.id,
            extractVersion: '^v(?<version>\\S+)',
        },
    },
    hivemind: {
        misePluginUrl: 'https://mise.jdx.dev/registry.html#tools',
        config: {
            packageName: 'DarthSim/hivemind',
            datasource: github_releases_1.GithubReleasesDatasource.id,
            extractVersion: '^v(?<version>\\S+)',
        },
    },
    jq: {
        misePluginUrl: 'https://mise.jdx.dev/registry.html#tools',
        config: {
            packageName: 'jqlang/jq',
            datasource: github_releases_1.GithubReleasesDatasource.id,
            extractVersion: '^jq-v(?<version>\\S+)',
        },
    },
    hk: {
        misePluginUrl: 'https://mise.jdx.dev/registry.html#tools',
        config: {
            packageName: 'jdx/hk',
            datasource: github_releases_1.GithubReleasesDatasource.id,
            extractVersion: '^v(?<version>\\S+)',
        },
    },
    kafka: {
        misePluginUrl: 'https://mise.jdx.dev/registry.html#tools',
        config: (version) => {
            const apacheMatches = /^apache-(?<version>\d\S+)/.exec(version)?.groups;
            if (apacheMatches) {
                return {
                    datasource: github_tags_1.GithubTagsDatasource.id,
                    packageName: 'apache/kafka',
                    currentValue: apacheMatches.version,
                };
            }
            return undefined;
        },
    },
    lefthook: {
        misePluginUrl: 'https://mise.jdx.dev/registry.html#tools',
        config: {
            packageName: 'evilmartians/lefthook',
            datasource: github_releases_1.GithubReleasesDatasource.id,
            extractVersion: '^v(?<version>\\S+)',
        },
    },
    localstack: {
        misePluginUrl: 'https://mise.jdx.dev/registry.html#tools',
        config: {
            packageName: 'localstack/localstack',
            datasource: github_releases_1.GithubReleasesDatasource.id,
            extractVersion: '^v(?<version>\\S+)',
        },
    },
    opentofu: {
        misePluginUrl: 'https://mise.jdx.dev/registry.html#tools',
        config: {
            packageName: 'opentofu/opentofu',
            datasource: github_releases_1.GithubReleasesDatasource.id,
            extractVersion: '^v(?<version>\\S+)',
        },
    },
    protoc: {
        misePluginUrl: 'https://mise.jdx.dev/registry.html#tools',
        config: {
            packageName: 'protocolbuffers/protobuf',
            datasource: github_releases_1.GithubReleasesDatasource.id,
            extractVersion: '^v(?<version>\\S+)',
        },
    },
    redis: {
        misePluginUrl: 'https://mise.jdx.dev/registry.html#tools',
        config: {
            packageName: 'redis/redis',
            datasource: github_releases_1.GithubReleasesDatasource.id,
        },
    },
    ruff: {
        misePluginUrl: 'https://mise.jdx.dev/registry.html#tools',
        config: {
            packageName: 'astral-sh/ruff',
            datasource: github_releases_1.GithubReleasesDatasource.id,
        },
    },
    shellcheck: {
        misePluginUrl: 'https://mise.jdx.dev/registry.html#tools',
        config: {
            packageName: 'koalaman/shellcheck',
            datasource: github_releases_1.GithubReleasesDatasource.id,
            extractVersion: '^v(?<version>\\S+)',
        },
    },
    skeema: {
        misePluginUrl: 'https://mise.jdx.dev/registry.html#tools',
        config: {
            packageName: 'skeema/skeema',
            datasource: github_releases_1.GithubReleasesDatasource.id,
            extractVersion: '^v(?<version>\\S+)',
        },
    },
    sops: {
        misePluginUrl: 'https://mise.jdx.dev/registry.html#tools',
        config: {
            packageName: 'getsops/sops',
            datasource: github_releases_1.GithubReleasesDatasource.id,
            extractVersion: '^v(?<version>\\S+)',
        },
    },
    stripe: {
        misePluginUrl: 'https://mise.jdx.dev/registry.html#tools',
        config: {
            packageName: 'stripe/stripe-cli',
            datasource: github_releases_1.GithubReleasesDatasource.id,
            extractVersion: '^v(?<version>\\S+)',
        },
    },
    terragrunt: {
        misePluginUrl: 'https://mise.jdx.dev/registry.html#tools',
        config: {
            packageName: 'gruntwork-io/terragrunt',
            datasource: github_releases_1.GithubReleasesDatasource.id,
            extractVersion: '^v(?<version>\\S+)',
        },
    },
    tilt: {
        misePluginUrl: 'https://mise.jdx.dev/registry.html#tools',
        config: {
            packageName: 'tilt-dev/tilt',
            datasource: github_releases_1.GithubReleasesDatasource.id,
            extractVersion: '^v(?<version>\\S+)',
        },
    },
    tusd: {
        misePluginUrl: 'https://mise.jdx.dev/registry.html#tools',
        config: {
            packageName: 'tus/tusd',
            datasource: github_releases_1.GithubReleasesDatasource.id,
            extractVersion: '^v(?<version>\\S+)',
        },
    },
};
exports.miseTooling = {
    ...miseCoreTooling,
    ...miseRegistryTooling,
};
//# sourceMappingURL=upgradeable-tooling.js.map