"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.extractPackageFile = extractPackageFile;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const logger_1 = require("../../../logger");
const regex_1 = require("../../../util/regex");
const backends_1 = require("./backends");
const upgradeable_tooling_1 = require("./upgradeable-tooling");
const utils_1 = require("./utils");
// Tool names can have options in the tool name
// e.g. ubi:tamasfe/taplo[matching=full,exe=taplo]
const optionInToolNameRegex = (0, regex_1.regEx)(/^(?<name>.+?)(?:\[(?<options>.+)\])?$/);
function extractPackageFile(content, packageFile) {
    logger_1.logger.trace(`mise.extractPackageFile(${packageFile})`);
    const misefile = (0, utils_1.parseTomlFile)(content, packageFile);
    if (!misefile) {
        return null;
    }
    const deps = [];
    const tools = misefile.tools;
    if (tools) {
        for (const [name, toolData] of Object.entries(tools)) {
            const version = parseVersion(toolData);
            // Parse the tool options in the tool name
            const { name: depName, options: optionsInName } = optionInToolNameRegex.exec(name.trim()).groups;
            const delimiterIndex = name.indexOf(':');
            const backend = depName.substring(0, delimiterIndex);
            const toolName = depName.substring(delimiterIndex + 1);
            const options = parseOptions(optionsInName, is_1.default.nonEmptyObject(toolData) ? toolData : {});
            const toolConfig = version === null
                ? null
                : getToolConfig(backend, toolName, version, options);
            const dep = createDependency(depName, version, toolConfig);
            deps.push(dep);
        }
    }
    return deps.length ? { deps } : null;
}
function parseVersion(toolData) {
    if (is_1.default.nonEmptyString(toolData)) {
        // Handle the string case
        // e.g. 'erlang = "23.3"'
        return toolData;
    }
    if (is_1.default.array(toolData, is_1.default.string)) {
        // Handle the array case
        // e.g. 'erlang = ["23.3", "24.0"]'
        return toolData.length ? toolData[0] : null; // Get the first version in the array
    }
    if (is_1.default.object(toolData) && is_1.default.nonEmptyString(toolData.version)) {
        // Handle the object case with a string version
        // e.g. 'python = { version = "3.11.2" }'
        return toolData.version;
    }
    return null; // Return null if no version is found
}
function parseOptions(optionsInName, toolOptions) {
    const options = is_1.default.nonEmptyString(optionsInName)
        ? Object.fromEntries(optionsInName.split(',').map((option) => option.split('=', 2)))
        : {};
    // Options in toolOptions will override options in the tool name
    return {
        ...options,
        ...toolOptions,
    };
}
function getToolConfig(backend, toolName, version, toolOptions) {
    switch (backend) {
        case '':
            // If the tool name does not specify a backend, it should be a short name or an alias defined by users
            return getRegistryToolConfig(toolName, version);
        // We can specify core, asdf, vfox, aqua backends for tools in the default registry
        // e.g. 'core:rust', 'asdf:rust', 'vfox:clang', 'aqua:act'
        case 'core':
            return getConfigFromTooling(upgradeable_tooling_1.miseTooling, toolName, version);
        case 'asdf':
            return getConfigFromTooling(upgradeable_tooling_1.asdfTooling, toolName, version);
        case 'vfox':
            return getRegistryToolConfig(toolName, version);
        case 'aqua':
            return (getRegistryToolConfig(toolName, version) ??
                (0, backends_1.createAquaToolConfig)(toolName, version));
        case 'cargo':
            return (0, backends_1.createCargoToolConfig)(toolName, version);
        case 'dotnet':
            return (0, backends_1.createDotnetToolConfig)(toolName);
        case 'gem':
            return (0, backends_1.createGemToolConfig)(toolName);
        case 'go':
            return (0, backends_1.createGoToolConfig)(toolName);
        case 'npm':
            return (0, backends_1.createNpmToolConfig)(toolName);
        case 'pipx':
            return (0, backends_1.createPipxToolConfig)(toolName);
        case 'spm':
            return (0, backends_1.createSpmToolConfig)(toolName);
        case 'ubi':
            return (0, backends_1.createUbiToolConfig)(toolName, version, toolOptions);
        default:
            // Unsupported backend
            return null;
    }
}
/**
 * Get the tooling config for a short name defined in the default registry
 * @link https://mise.jdx.dev/registry.html
 */
function getRegistryToolConfig(short, version) {
    // Try to get the config from miseTooling first, then asdfTooling
    return (getConfigFromTooling(upgradeable_tooling_1.miseTooling, short, version) ??
        getConfigFromTooling(upgradeable_tooling_1.asdfTooling, short, version));
}
function getConfigFromTooling(toolingSource, name, version) {
    const toolDefinition = toolingSource[name];
    if (!toolDefinition) {
        return null;
    } // Return null if no toolDefinition is found
    return ((is_1.default.function(toolDefinition.config)
        ? toolDefinition.config(version)
        : toolDefinition.config) ?? null); // Ensure null is returned instead of undefined
}
function createDependency(name, version, config) {
    if (version === null) {
        return {
            depName: name,
            skipReason: 'unspecified-version',
        };
    }
    if (config === null) {
        return {
            depName: name,
            skipReason: 'unsupported-datasource',
        };
    }
    return {
        depName: name,
        currentValue: version,
        // Spread the config last to override other properties
        ...config,
    };
}
//# sourceMappingURL=extract.js.map