"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.parseRepository = parseRepository;
exports.resolveAlias = resolveAlias;
exports.getRepositories = getRepositories;
exports.isAlias = isAlias;
exports.aliasRecordToRepositories = aliasRecordToRepositories;
exports.isFileInDir = isFileInDir;
const tslib_1 = require("tslib");
const upath_1 = tslib_1.__importDefault(require("upath"));
const logger_1 = require("../../../logger");
const docker_1 = require("../../datasource/docker");
const oci_1 = require("./oci");
function parseRepository(depName, repositoryURL) {
    const res = {};
    try {
        const url = new URL(repositoryURL);
        switch (url.protocol) {
            case 'oci:':
                res.datasource = docker_1.DockerDatasource.id;
                res.packageName = `${(0, oci_1.removeOCIPrefix)(repositoryURL)}/${depName}`;
                // https://github.com/helm/helm/issues/10312
                // https://github.com/helm/helm/issues/10678
                res.pinDigests = false;
                break;
            case 'file:':
                res.skipReason = 'local-dependency';
                break;
            default:
                res.registryUrls = [repositoryURL];
        }
    }
    catch (err) {
        logger_1.logger.debug({ err }, 'Error parsing url');
        res.skipReason = 'invalid-url';
    }
    return res;
}
/**
 * Resolves alias in repository string.
 *
 * @param repository to be resolved string
 * @param registryAliases Records containing registryAliases as key and to be resolved URLs as values
 *
 * @returns  resolved alias. If repository does not contain an alias the repository string will be returned. Should it contain an alias which can not be resolved using `registryAliases`, null will be returned
 */
function resolveAlias(repository, registryAliases) {
    if (!isAlias(repository)) {
        return repository;
    }
    const repoWithPrefixRemoved = repository.slice(repository.startsWith('@') ? 1 : 6);
    const alias = registryAliases[repoWithPrefixRemoved];
    if (alias) {
        return alias;
    }
    return null;
}
function getRepositories(definitions) {
    const repositoryList = definitions
        .flatMap((value) => value.dependencies)
        .filter((dependency) => dependency.repository) // only keep non-local references --> if no repository is defined the chart will be searched in charts/<name>
        .filter((dependency) => !isAlias(dependency.repository)) // do not add registryAliases
        .filter((dependency) => !dependency.repository.startsWith('file:')) // skip repositories which are locally referenced
        .map((dependency) => {
        // remove additional keys to prevent interference at deduplication
        return {
            name: dependency.name,
            repository: dependency.repository,
        };
    });
    const dedup = new Set();
    return repositoryList.filter((el) => {
        const duplicate = dedup.has(el.repository);
        dedup.add(el.repository);
        return !duplicate;
    });
}
function isAlias(repository) {
    if (!repository) {
        return false;
    }
    return repository.startsWith('@') || repository.startsWith('alias:');
}
function aliasRecordToRepositories(registryAliases) {
    return Object.entries(registryAliases)
        .filter(([, url]) => /^(https?|oci):\/\/.+/.exec(url))
        .map(([alias, url]) => {
        return {
            name: alias,
            repository: url,
        };
    });
}
function isFileInDir(dir, file) {
    return upath_1.default.dirname(file) === dir;
}
//# sourceMappingURL=utils.js.map