"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.findDependencies = findDependencies;
exports.findDependenciesInternal = findDependenciesInternal;
exports.extractPackageFile = extractPackageFile;
const logger_1 = require("../../../logger");
const yaml_1 = require("../../../util/yaml");
const docker_1 = require("../../versioning/docker");
const extract_1 = require("../dockerfile/extract");
const util_1 = require("./util");
function getHelmDep(registry, repository, tag, registryAliases) {
    const dep = (0, extract_1.getDep)(`${registry}${repository}:${tag}`, false, registryAliases);
    dep.replaceString = tag;
    dep.versioning = docker_1.id;
    dep.autoReplaceStringTemplate =
        '{{newValue}}{{#if newDigest}}@{{newDigest}}{{/if}}';
    return dep;
}
/**
 * Recursively find all supported dependencies in the yaml object.
 *
 * @param parsedContent
 */
function findDependencies(parsedContent, registryAliases) {
    return findDependenciesInternal(parsedContent, [], registryAliases);
}
function findDependenciesInternal(parsedContent, packageDependencies, registryAliases) {
    if (!parsedContent || typeof parsedContent !== 'object') {
        return packageDependencies;
    }
    Object.entries(parsedContent).forEach(([key, value]) => {
        if ((0, util_1.matchesHelmValuesDockerHeuristic)(key, value)) {
            const currentItem = value;
            let registry = currentItem.registry;
            registry = registry ? `${registry}/` : '';
            const repository = String(currentItem.repository);
            const tag = `${currentItem.tag ?? currentItem.version}`;
            packageDependencies.push(getHelmDep(registry, repository, tag, registryAliases));
        }
        else if ((0, util_1.matchesHelmValuesInlineImage)(key, value)) {
            packageDependencies.push((0, extract_1.getDep)(value, true, registryAliases));
        }
        else {
            findDependenciesInternal(value, packageDependencies, registryAliases);
        }
    });
    return packageDependencies;
}
function extractPackageFile(content, packageFile, config) {
    let parsedContent;
    try {
        // a parser that allows extracting line numbers would be preferable, with
        // the current approach we need to match anything we find again during the update
        // TODO: fix me (#9610)
        parsedContent = (0, yaml_1.parseYaml)(content);
    }
    catch (err) {
        logger_1.logger.debug({ err, packageFile }, 'Failed to parse helm-values YAML');
        return null;
    }
    try {
        const deps = [];
        for (const con of parsedContent) {
            deps.push(...findDependencies(con, config.registryAliases));
        }
        if (deps.length) {
            return { deps };
        }
    }
    catch (err) /* istanbul ignore next */ {
        logger_1.logger.debug({ err, packageFile }, 'Error parsing helm-values parsed content');
    }
    return null;
}
//# sourceMappingURL=extract.js.map