"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.versionLikeSubstring = versionLikeSubstring;
exports.isDependencyString = isDependencyString;
exports.parseDependencyString = parseDependencyString;
exports.isGradleScriptFile = isGradleScriptFile;
exports.isGradleVersionsFile = isGradleVersionsFile;
exports.isGradleBuildFile = isGradleBuildFile;
exports.isPropsFile = isPropsFile;
exports.isKotlinSourceFile = isKotlinSourceFile;
exports.isTOMLFile = isTOMLFile;
exports.toAbsolutePath = toAbsolutePath;
exports.reorderFiles = reorderFiles;
exports.getVars = getVars;
exports.updateVars = updateVars;
const tslib_1 = require("tslib");
const upath_1 = tslib_1.__importDefault(require("upath"));
const regex_1 = require("../../../util/regex");
const gradle_1 = require("../../versioning/gradle");
const artifactRegex = (0, regex_1.regEx)('^[a-zA-Z][-_a-zA-Z0-9]*(?:\\.[a-zA-Z0-9][-_a-zA-Z0-9]*?)*$');
const versionLikeRegex = (0, regex_1.regEx)('^(?<version>[-_.\\[\\](),a-zA-Z0-9+ ]+)');
// Extracts version-like and range-like strings from the beginning of input
function versionLikeSubstring(input) {
    if (!input) {
        return null;
    }
    const match = versionLikeRegex.exec(input);
    const version = match?.groups?.version?.trim();
    if (!version || !(0, regex_1.regEx)(/\d/).test(version)) {
        return null;
    }
    if (!gradle_1.api.isValid(version)) {
        return null;
    }
    return version;
}
function isDependencyString(input) {
    const [depNotation, ...extra] = input.split('@');
    if (extra.length > 1) {
        return false;
    }
    const parts = depNotation.split(':');
    if (parts.length !== 3 && parts.length !== 4) {
        return false;
    }
    const [groupId, artifactId, version, classifier] = parts;
    return !!(groupId &&
        artifactId &&
        version &&
        artifactRegex.test(groupId) &&
        artifactRegex.test(artifactId) &&
        (!classifier || artifactRegex.test(classifier)) &&
        version === versionLikeSubstring(version));
}
function parseDependencyString(input) {
    if (!isDependencyString(input)) {
        return null;
    }
    const [depNotation, dataType] = input.split('@');
    const [groupId, artifactId, currentValue] = depNotation.split(':');
    return {
        depName: `${groupId}:${artifactId}`,
        currentValue,
        ...(dataType && { dataType }),
    };
}
const gradleVersionsFileRegex = (0, regex_1.regEx)('^versions\\.gradle(?:\\.kts)?$', 'i');
const gradleBuildFileRegex = (0, regex_1.regEx)('^build\\.gradle(?:\\.kts)?$', 'i');
function isGradleScriptFile(path) {
    const filename = upath_1.default.basename(path).toLowerCase();
    return filename.endsWith('.gradle.kts') || filename.endsWith('.gradle');
}
function isGradleVersionsFile(path) {
    const filename = upath_1.default.basename(path);
    return gradleVersionsFileRegex.test(filename);
}
function isGradleBuildFile(path) {
    const filename = upath_1.default.basename(path);
    return gradleBuildFileRegex.test(filename);
}
function isPropsFile(path) {
    const filename = upath_1.default.basename(path).toLowerCase();
    return filename === 'gradle.properties';
}
function isKotlinSourceFile(path) {
    const filename = upath_1.default.basename(path).toLowerCase();
    return filename.endsWith('.kt');
}
function isTOMLFile(path) {
    const filename = upath_1.default.basename(path).toLowerCase();
    return filename.endsWith('.toml');
}
function toAbsolutePath(packageFile) {
    return upath_1.default.join(packageFile.replace((0, regex_1.regEx)(/^[/\\]*/), '/'));
}
function getFileRank(filename) {
    if (isPropsFile(filename)) {
        return 0;
    }
    if (isGradleVersionsFile(filename)) {
        return 1;
    }
    if (isGradleBuildFile(filename)) {
        return 3;
    }
    return 2;
}
function reorderFiles(packageFiles) {
    return packageFiles.sort((x, y) => {
        const xAbs = toAbsolutePath(x);
        const yAbs = toAbsolutePath(y);
        const xDir = upath_1.default.dirname(xAbs);
        const yDir = upath_1.default.dirname(yAbs);
        if (xDir === yDir) {
            const xRank = getFileRank(xAbs);
            const yRank = getFileRank(yAbs);
            if (xRank === yRank) {
                if (xAbs > yAbs) {
                    return 1;
                }
                if (xAbs < yAbs) {
                    return -1;
                }
            }
            else if (xRank > yRank) {
                return 1;
            }
            else if (yRank > xRank) {
                return -1;
            }
        }
        else if (xDir.startsWith(yDir)) {
            return 1;
        }
        else if (yDir.startsWith(xDir)) {
            return -1;
        }
        return 0;
    });
}
function getVars(registry, dir, vars = registry[dir] || {}) {
    const dirAbs = toAbsolutePath(dir);
    const parentDir = upath_1.default.dirname(dirAbs);
    if (parentDir === dirAbs) {
        return vars;
    }
    const parentVars = registry[parentDir] || {};
    return getVars(registry, parentDir, { ...parentVars, ...vars });
}
function updateVars(registry, dir, newVars) {
    const oldVars = registry[dir] ?? {};
    registry[dir] = { ...oldVars, ...newVars };
}
//# sourceMappingURL=utils.js.map