"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.extraEnv = void 0;
exports.gradleWrapperFileName = gradleWrapperFileName;
exports.prepareGradleCommand = prepareGradleCommand;
exports.getJavaConstraint = getJavaConstraint;
exports.getJvmConfiguration = getJvmConfiguration;
exports.getJavaLanguageVersion = getJavaLanguageVersion;
exports.extractGradleVersion = extractGradleVersion;
const tslib_1 = require("tslib");
const node_os_1 = tslib_1.__importDefault(require("node:os"));
const upath_1 = require("upath");
const global_1 = require("../../../config/global");
const logger_1 = require("../../../logger");
const fs_1 = require("../../../util/fs");
const regex_1 = require("../../../util/regex");
const gradle_1 = tslib_1.__importDefault(require("../../versioning/gradle"));
const parser_1 = require("../gradle/parser");
exports.extraEnv = {
    GRADLE_OPTS: '-Dorg.gradle.parallel=true -Dorg.gradle.configureondemand=true -Dorg.gradle.daemon=false -Dorg.gradle.caching=false',
};
function gradleWrapperFileName() {
    if (node_os_1.default.platform() === 'win32' &&
        global_1.GlobalConfig.get('binarySource') !== 'docker') {
        return 'gradlew.bat';
    }
    return './gradlew';
}
async function prepareGradleCommand(gradlewFile) {
    const gradlewStat = await (0, fs_1.statLocalFile)(gradlewFile);
    if (gradlewStat?.isFile() === true) {
        // if the file is not executable by others
        if (node_os_1.default.platform() !== 'win32' && (gradlewStat.mode & 0o1) === 0) {
            logger_1.logger.debug('Gradle wrapper is missing the executable bit');
            // add the execution permission to the owner, group and others
            await (0, fs_1.chmodLocalFile)(gradlewFile, gradlewStat.mode | 0o111);
        }
        return gradleWrapperFileName();
    }
    return null;
}
/**
 * Find compatible java version for gradle.
 * see https://docs.gradle.org/current/userguide/compatibility.html
 * @param gradleVersion current gradle version
 * @param gradlewFile path to gradle wrapper
 * @returns A Java semver range
 */
async function getJavaConstraint(gradleVersion, gradlewFile) {
    const major = gradleVersion ? gradle_1.default.getMajor(gradleVersion) : null;
    const minor = gradleVersion ? gradle_1.default.getMinor(gradleVersion) : null;
    if (major) {
        // https://docs.gradle.org/8.8/release-notes.html#daemon-toolchains
        if (major > 8 || (major === 8 && minor && minor >= 8)) {
            const toolChainVersion = await getJvmConfiguration(gradlewFile);
            if (toolChainVersion) {
                return `^${toolChainVersion}.0.0`;
            }
        }
        // https://docs.gradle.org/6.7/release-notes.html#new-jvm-ecosystem-features
        if (major > 6 || (major === 6 && minor && minor >= 7)) {
            const languageVersion = await getJavaLanguageVersion(gradlewFile);
            if (languageVersion) {
                return `^${languageVersion}.0.0`;
            }
        }
        if (major > 8 || (major === 8 && minor && minor >= 5)) {
            return '^21.0.0';
        }
        if (major > 7 || (major === 7 && minor && minor >= 3)) {
            return '^17.0.0';
        }
        if (major === 7) {
            return '^16.0.0';
        }
        // first public gradle version was 2.0
        if (major > 0 && major < 5) {
            return '^8.0.0';
        }
    }
    return '^11.0.0';
}
/**
 * https://docs.gradle.org/current/userguide/gradle_daemon.html#sec:daemon_jvm_criteria
 */
async function getJvmConfiguration(gradlewFile) {
    const daemonJvmFile = (0, upath_1.join)((0, upath_1.dirname)(gradlewFile), 'gradle/gradle-daemon-jvm.properties');
    const daemonJvm = await (0, fs_1.readLocalFile)(daemonJvmFile, 'utf8');
    if (daemonJvm) {
        const TOOLCHAIN_VERSION_REGEX = (0, regex_1.regEx)('^(?:toolchainVersion\\s*=\\s*)(?<version>\\d+)$', 'm');
        const toolChainMatch = TOOLCHAIN_VERSION_REGEX.exec(daemonJvm);
        if (toolChainMatch?.groups) {
            return toolChainMatch.groups.version;
        }
    }
    return null;
}
/**
 * https://docs.gradle.org/current/userguide/toolchains.html#sec:consuming
 */
async function getJavaLanguageVersion(gradlewFile) {
    const localGradleDir = (0, upath_1.dirname)(gradlewFile);
    let buildFileName = (0, upath_1.join)(localGradleDir, 'build.gradle');
    if (!(await (0, fs_1.localPathExists)(buildFileName))) {
        buildFileName = (0, upath_1.join)(localGradleDir, 'build.gradle.kts');
    }
    const buildFileContent = await (0, fs_1.readLocalFile)(buildFileName, 'utf8');
    if (!buildFileContent) {
        logger_1.logger.debug('build.gradle or build.gradle.kts not found');
        return null;
    }
    return (0, parser_1.parseJavaToolchainVersion)(buildFileContent);
}
// https://regex101.com/r/IcOs7P/1
const DISTRIBUTION_URL_REGEX = (0, regex_1.regEx)('^(?:distributionUrl\\s*=\\s*)(?<url>\\S*-(?<version>\\d+\\.\\d+(?:\\.\\d+)?(?:-\\w+)*)-(?<type>bin|all)\\.zip)\\s*$', 'm');
function extractGradleVersion(fileContent) {
    const distributionUrlMatch = DISTRIBUTION_URL_REGEX.exec(fileContent);
    if (distributionUrlMatch?.groups) {
        return {
            url: distributionUrlMatch.groups.url,
            version: distributionUrlMatch.groups.version,
        };
    }
    logger_1.logger.debug('Gradle wrapper version and url could not be extracted from properties - skipping update');
    return null;
}
//# sourceMappingURL=utils.js.map