"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.endBlockRegex = exports.excludeBlockStartRegex = void 0;
exports.parseLine = parseLine;
const tslib_1 = require("tslib");
const semver_1 = tslib_1.__importDefault(require("semver"));
const regex_1 = require("../../../util/regex");
const go_1 = require("../../datasource/go");
const golang_version_1 = require("../../datasource/golang-version");
const semver_2 = require("../../versioning/semver");
function trimQuotes(str) {
    return str.replace((0, regex_1.regEx)(/^"(.*)"$/), '$1');
}
const requireRegex = (0, regex_1.regEx)(/^(?<keyword>require)?\s+(?<module>[^\s]+\/?[^\s]+)\s+(?<version>[^\s]+)(?:\s*\/\/\s*(?<comment>[^\s]+)\s*)?$/);
const replaceRegex = (0, regex_1.regEx)(/^(?<keyword>replace)?\s+(?<module>[^\s]+\/[^\s]+)\s*=>\s*(?<replacement>[^\s]+)(?:\s+(?<version>[^\s]+))?(?:\s*\/\/\s*(?<comment>[^\s]+)\s*)?$/);
exports.excludeBlockStartRegex = (0, regex_1.regEx)(/^(?<keyword>exclude)\s+\(\s*$/);
exports.endBlockRegex = (0, regex_1.regEx)(/^\s+\)\s*$/);
const toolRegex = (0, regex_1.regEx)(/^(?<keyword>tool)?\s+(?<module>[^\s]+\/?[^\s]+)\s*$/);
const goVersionRegex = (0, regex_1.regEx)(/^\s*go\s+(?<version>[^\s]+)\s*$/);
const toolchainVersionRegex = (0, regex_1.regEx)(/^\s*toolchain\s+go(?<version>[^\s]+)\s*$/);
const pseudoVersionRegex = (0, regex_1.regEx)(go_1.GoDatasource.pversionRegexp);
function extractDigest(input) {
    const match = pseudoVersionRegex.exec(input);
    return match?.groups?.digest;
}
function parseLine(input) {
    const goVersionMatches = goVersionRegex.exec(input)?.groups;
    if (goVersionMatches) {
        const { version: currentValue } = goVersionMatches;
        const dep = {
            datasource: golang_version_1.GolangVersionDatasource.id,
            versioning: 'go-mod-directive',
            depType: 'golang',
            depName: 'go',
            currentValue,
        };
        if (!semver_1.default.validRange(currentValue)) {
            dep.skipReason = 'invalid-version';
        }
        return dep;
    }
    const toolchainMatches = toolchainVersionRegex.exec(input)?.groups;
    if (toolchainMatches) {
        const { version: currentValue } = toolchainMatches;
        const dep = {
            datasource: golang_version_1.GolangVersionDatasource.id,
            depType: 'toolchain',
            depName: 'go',
            currentValue,
        };
        if (!semver_1.default.valid(currentValue)) {
            dep.skipReason = 'invalid-version';
        }
        return dep;
    }
    const requireMatches = requireRegex.exec(input)?.groups;
    if (requireMatches) {
        const { keyword, module, version: currentValue, comment } = requireMatches;
        const depName = trimQuotes(module);
        const dep = {
            datasource: go_1.GoDatasource.id,
            depType: 'require',
            depName,
            currentValue,
        };
        if ((0, semver_2.isVersion)(currentValue)) {
            const digest = extractDigest(currentValue);
            if (digest) {
                dep.currentDigest = digest;
                dep.digestOneAndOnly = true;
                dep.versioning = 'loose';
            }
        }
        else {
            dep.skipReason = 'invalid-version';
        }
        if (comment === 'indirect') {
            dep.depType = 'indirect';
            dep.enabled = false;
        }
        if (!keyword) {
            dep.managerData = { multiLine: true };
        }
        return dep;
    }
    const replaceMatches = replaceRegex.exec(input)?.groups;
    if (replaceMatches) {
        const { keyword, replacement, version: currentValue, comment, } = replaceMatches;
        const depName = trimQuotes(replacement);
        const dep = {
            datasource: go_1.GoDatasource.id,
            depType: 'replace',
            depName,
            currentValue,
        };
        if ((0, semver_2.isVersion)(currentValue)) {
            const digest = extractDigest(currentValue);
            if (digest) {
                dep.currentDigest = digest;
                dep.digestOneAndOnly = true;
                dep.versioning = 'loose';
            }
        }
        else if (currentValue) {
            dep.skipReason = 'invalid-version';
        }
        else {
            dep.skipReason = 'unspecified-version';
            delete dep.currentValue;
        }
        if (comment === 'indirect') {
            dep.depType = 'indirect';
            dep.enabled = false;
        }
        if (!keyword) {
            dep.managerData = { multiLine: true };
        }
        if (depName.startsWith('/') || depName.startsWith('.')) {
            dep.skipReason = 'local-dependency';
        }
        return dep;
    }
    const toolMatches = toolRegex.exec(input)?.groups;
    if (toolMatches) {
        const { keyword, module } = toolMatches;
        const depName = trimQuotes(module);
        const dep = {
            datasource: go_1.GoDatasource.id,
            depType: 'tool',
            depName,
            skipReason: 'unversioned-reference',
        };
        if (!keyword) {
            dep.managerData = { multiLine: true };
        }
        return dep;
    }
    return null;
}
//# sourceMappingURL=line-parser.js.map