"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getExtraDeps = getExtraDeps;
exports.extraDepsTable = extraDepsTable;
exports.getExtraDepsNotice = getExtraDepsNotice;
const diff_1 = require("diff");
const markdown_table_1 = require("markdown-table");
const line_parser_1 = require("./line-parser");
function getExtraDeps(goModBefore, goModAfter, excludeDeps) {
    const result = [];
    const diff = (0, diff_1.diffLines)(goModBefore, goModAfter, {
        newlineIsToken: true,
    });
    const addDeps = {};
    const rmDeps = {};
    for (const { added, removed, value } of diff) {
        if (!added && !removed) {
            continue;
        }
        const res = (0, line_parser_1.parseLine)(value);
        if (!res) {
            continue;
        }
        const { depName, depType, currentValue } = res;
        if (!depName || !currentValue) {
            continue;
        }
        let expandedDepName = depName;
        // NOTE: Right now the only special depType we care about is 'toolchain' because the table
        // rendering prior to this change was ambiguous with regards to go version vs toolchain
        // version updates.
        if (depType === 'toolchain') {
            expandedDepName = `${depName} (${depType})`;
        }
        if (added) {
            addDeps[expandedDepName] = currentValue;
        }
        else {
            rmDeps[expandedDepName] = currentValue;
        }
    }
    for (const [depName, currentValue] of Object.entries(rmDeps)) {
        if (excludeDeps.includes(depName)) {
            continue;
        }
        const newValue = addDeps[depName];
        if (newValue) {
            result.push({
                depName,
                currentValue,
                newValue,
            });
        }
    }
    return result;
}
function extraDepsTable(extraDeps) {
    const tableLines = [];
    tableLines.push(['**Package**', '**Change**']);
    for (const { depName, currentValue, newValue } of extraDeps) {
        const depNameQuoted = `\`${depName}\``;
        const versionChangeQuoted = `\`${currentValue}\` -> \`${newValue}\``;
        tableLines.push([depNameQuoted, versionChangeQuoted]);
    }
    return (0, markdown_table_1.markdownTable)(tableLines, {
        align: ['l', 'l'],
    });
}
function getExtraDepsNotice(goModBefore, goModAfter, excludeDeps) {
    if (!goModBefore || !goModAfter) {
        return null;
    }
    const extraDeps = getExtraDeps(goModBefore, goModAfter, excludeDeps);
    if (extraDeps.length === 0) {
        return null;
    }
    const noticeLines = [
        'In order to perform the update(s) described in the table above, Renovate ran the `go get` command, which resulted in the following additional change(s):',
        '\n',
    ];
    const goUpdated = extraDeps.some(({ depName }) => depName === 'go');
    const toolchainUpdated = extraDeps.some(({ depName }) => depName === 'go (toolchain)');
    const otherDepsCount = extraDeps.length - (goUpdated ? 1 : 0) - (toolchainUpdated ? 1 : 0);
    if (otherDepsCount === 1) {
        noticeLines.push(`- ${otherDepsCount} additional dependency was updated`);
    }
    else if (otherDepsCount > 1) {
        noticeLines.push(`- ${otherDepsCount} additional dependencies were updated`);
    }
    if (goUpdated) {
        noticeLines.push('- The `go` directive was updated for compatibility reasons');
    }
    noticeLines.push('\n');
    noticeLines.push('Details:');
    noticeLines.push('\n');
    noticeLines.push(extraDepsTable(extraDeps));
    return noticeLines.join('\n');
}
//# sourceMappingURL=artifacts-extra.js.map