"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.extractPackageFile = extractPackageFile;
const logger_1 = require("../../../logger");
const array_1 = require("../../../util/array");
const fs_1 = require("../../../util/fs");
const hex_1 = require("../../datasource/hex");
const hex_2 = require("../../versioning/hex");
const locked_version_1 = require("./locked-version");
const schema_1 = require("./schema");
const dependencySections = ['dependencies', 'dev-dependencies'];
// map gleam.toml section keys to standard renovate depType's
// this allows us to leverage renovate built-in configurations and presets
function mapSectionKey(sectionKey) {
    switch (sectionKey) {
        case 'dev-dependencies':
            return 'devDependencies';
        default:
            return sectionKey;
    }
}
function toPackageDep({ name, sectionKey, version, }) {
    return {
        depName: name,
        depType: mapSectionKey(sectionKey),
        datasource: hex_1.HexDatasource.id,
        currentValue: version,
    };
}
function toPackageDeps({ deps, sectionKey, }) {
    return Object.entries(deps ?? {}).map(([name, version]) => toPackageDep({ name, sectionKey, version }));
}
function extractGleamTomlDeps(gleamToml) {
    return dependencySections.flatMap((sectionKey) => toPackageDeps({
        deps: gleamToml[sectionKey],
        sectionKey,
    }));
}
async function extractPackageFile(content, packageFile) {
    const result = schema_1.GleamToml.safeParse(content);
    if (!result.success) {
        logger_1.logger.debug({ err: result.error, packageFile }, 'Error parsing Gleam package file content');
        return null;
    }
    const deps = extractGleamTomlDeps(result.data);
    if (!deps.length) {
        logger_1.logger.debug(`No dependencies found in Gleam package file ${packageFile}`);
        return null;
    }
    const packageFileContent = { deps };
    const lockFileName = (0, fs_1.getSiblingFileName)(packageFile, 'manifest.toml');
    const lockFileExists = await (0, fs_1.localPathExists)(lockFileName);
    if (!lockFileExists) {
        logger_1.logger.debug(`Lock file ${lockFileName} does not exist.`);
        return packageFileContent;
    }
    const versionsByPackage = await (0, locked_version_1.extractLockFileVersions)(lockFileName);
    if (!versionsByPackage) {
        return packageFileContent;
    }
    packageFileContent.lockFiles = [lockFileName];
    for (const dep of packageFileContent.deps) {
        const packageName = dep.depName;
        const versions = (0, array_1.coerceArray)(versionsByPackage.get(packageName));
        const lockedVersion = hex_2.api.getSatisfyingVersion(versions, dep.currentValue);
        if (lockedVersion) {
            dep.lockedVersion = lockedVersion;
        }
        else {
            logger_1.logger.debug(`No locked version found for package ${dep.depName} in the range of ${dep.currentValue}.`);
        }
    }
    return packageFileContent;
}
//# sourceMappingURL=extract.js.map