"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.extractPackageFile = extractPackageFile;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const regex_1 = require("../../../util/regex");
const yaml_1 = require("../../../util/yaml");
const git_tags_1 = require("../../datasource/git-tags");
const helm_1 = require("../../datasource/helm");
const extract_1 = require("../dockerfile/extract");
const oci_1 = require("../helmv3/oci");
const util_1 = require("../terraform/util");
const schema_1 = require("./schema");
function extractGitRepo(doc) {
    const dep = {
        depType: 'git_repo',
        datasource: git_tags_1.GitTagsDatasource.id,
    };
    const repo = doc.spec?.repo;
    if (!repo) {
        return {
            ...dep,
            skipReason: 'missing-depname',
        };
    }
    dep.sourceUrl = repo;
    dep.depName = repo;
    const currentValue = doc.spec.revision;
    if (!currentValue) {
        return {
            ...dep,
            skipReason: 'unspecified-version',
        };
    }
    return {
        ...dep,
        currentValue,
    };
}
function extractFleetHelmBlock(doc) {
    const dep = {
        depType: 'fleet',
        datasource: helm_1.HelmDatasource.id,
    };
    if (!doc.chart) {
        return {
            ...dep,
            skipReason: 'missing-depname',
        };
    }
    if ((0, oci_1.isOCIRegistry)(doc.chart)) {
        const dockerDep = (0, extract_1.getDep)(`${(0, oci_1.removeOCIPrefix)(doc.chart)}:${doc.version}`, false);
        return {
            ...dockerDep,
            depType: 'fleet',
            // https://github.com/helm/helm/issues/10312
            // https://github.com/helm/helm/issues/10678
            pinDigests: false,
        };
    }
    dep.depName = doc.chart;
    dep.packageName = doc.chart;
    if (!doc.repo) {
        if ((0, util_1.checkIfStringIsPath)(doc.chart)) {
            return {
                ...dep,
                skipReason: 'local-chart',
            };
        }
        return {
            ...dep,
            skipReason: 'no-repository',
        };
    }
    dep.registryUrls = [doc.repo];
    const currentValue = doc.version;
    if (!doc.version) {
        return {
            ...dep,
            skipReason: 'unspecified-version',
        };
    }
    return {
        ...dep,
        currentValue,
    };
}
function extractFleetFile(doc) {
    const result = [];
    result.push(extractFleetHelmBlock(doc.helm));
    if (!is_1.default.undefined(doc.targetCustomizations)) {
        // remove version from helm block to allow usage of variables defined in the global block, but do not create PRs
        // if there is no version defined in the customization.
        const helmBlockContext = { ...doc.helm };
        delete helmBlockContext.version;
        for (const [index, custom] of doc.targetCustomizations.entries()) {
            const dep = extractFleetHelmBlock({
                // merge base config with customization
                ...helmBlockContext,
                ...custom.helm,
            });
            result.push({
                // overwrite name with customization name to allow splitting of PRs
                ...dep,
                depName: custom.name ?? `targetCustomization[${index}]`, // if no name is provided, use the index
            });
        }
    }
    return result;
}
function extractPackageFile(content, packageFile) {
    if (!content) {
        return null;
    }
    const deps = [];
    if ((0, regex_1.regEx)('fleet.ya?ml').test(packageFile)) {
        const docs = (0, yaml_1.parseYaml)(content, {
            customSchema: schema_1.FleetFile,
            failureBehaviour: 'filter',
        });
        const fleetDeps = docs.flatMap(extractFleetFile);
        deps.push(...fleetDeps);
    }
    else {
        const docs = (0, yaml_1.parseYaml)(content, {
            customSchema: schema_1.GitRepo,
            failureBehaviour: 'filter',
        });
        const gitRepoDeps = docs.flatMap(extractGitRepo);
        deps.push(...gitRepoDeps);
    }
    return deps.length ? { deps } : null;
}
//# sourceMappingURL=extract.js.map