"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.extractPackageFile = extractPackageFile;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const logger_1 = require("../../../logger");
const regex_1 = require("../../../util/regex");
const yaml_1 = require("../../../util/yaml");
const extract_1 = require("../dockerfile/extract");
const schema_1 = require("./schema");
class LineMapper {
    imageLines;
    constructor(content, filter) {
        this.imageLines = [...content.split(regex_1.newlineRegex).entries()]
            .filter((entry) => filter.test(entry[1]))
            .map(([lineNumber, line]) => ({ lineNumber, line, used: false }));
    }
    pluckLineNumber(imageName) {
        const lineMeta = this.imageLines.find(({ line, used }) => !used && imageName && line.includes(imageName));
        // istanbul ignore if
        if (!lineMeta) {
            return null;
        }
        lineMeta.used = true; // unset plucked lines so duplicates are skipped
        return lineMeta.lineNumber;
    }
}
function extractPackageFile(content, packageFile, extractConfig) {
    logger_1.logger.debug(`docker-compose.extractPackageFile(${packageFile})`);
    let config;
    try {
        config = (0, yaml_1.parseSingleYaml)(content, {
            customSchema: schema_1.DockerComposeFile,
            removeTemplates: true,
        });
    }
    catch (err) {
        logger_1.logger.debug({ err, packageFile }, `Parsing Docker Compose config YAML failed`);
        return null;
    }
    try {
        const lineMapper = new LineMapper(content, (0, regex_1.regEx)(/^\s*image:/));
        // docker-compose v1 places the services at the top level,
        // docker-compose v2+ places the services within a 'services' key
        // since docker-compose spec version 1.27, the 'version' key has
        // become optional and can no longer be used to differentiate
        // between v1 and v2.
        const services = config.services ?? config;
        const extensions = config.extensions ?? {};
        // Image name/tags for services are only eligible for update if they don't
        // use variables and if the image is not built locally
        const deps = Object.values(services || /* istanbul ignore next: can never happen */ {})
            .concat(Object.values(extensions))
            .filter((service) => is_1.default.string(service?.image) && !service?.build)
            .map((service) => {
            const dep = (0, extract_1.getDep)(service.image, true, extractConfig.registryAliases);
            const lineNumber = lineMapper.pluckLineNumber(service.image);
            // istanbul ignore if
            if (!lineNumber) {
                return null;
            }
            return dep;
        })
            .filter(is_1.default.truthy);
        logger_1.logger.trace({ deps }, 'Docker Compose image');
        return { deps };
    }
    catch (err) /* istanbul ignore next */ {
        logger_1.logger.debug({ packageFile, err }, 'Error extracting Docker Compose file');
        return null;
    }
}
//# sourceMappingURL=extract.js.map