"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.updateArtifacts = updateArtifacts;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const semver_1 = tslib_1.__importDefault(require("semver"));
const shlex_1 = require("shlex");
const logger_1 = require("../../../logger");
const exec_1 = require("../../../util/exec");
const fs_1 = require("../../../util/fs");
async function updateArtifacts({ config: { constraints, isLockFileMaintenance, updateType }, packageFileName, updatedDeps, }) {
    const lockFileName = (0, fs_1.getSiblingFileName)(packageFileName, 'devbox.lock');
    const existingLockFileContent = await (0, fs_1.readLocalFile)(lockFileName, 'utf8');
    if (!existingLockFileContent) {
        logger_1.logger.debug('No devbox.lock found');
        return null;
    }
    const supportsNoInstall = constraints?.devbox
        ? semver_1.default.intersects(constraints.devbox, '>=0.14.0')
        : true;
    const execOptions = {
        cwdFile: packageFileName,
        toolConstraints: [
            // we are required to install nix because devbox spawns nix commands internally
            // https://github.com/renovatebot/renovate/discussions/35382
            // https://github.com/jetify-com/devbox/issues/2585
            {
                toolName: 'nix',
                constraint: constraints?.nix,
            },
            {
                toolName: 'devbox',
                constraint: constraints?.devbox,
            },
        ],
        docker: {},
    };
    const cmd = [];
    if (isLockFileMaintenance) {
        cmd.push(supportsNoInstall ? 'devbox update --no-install' : 'devbox update');
    }
    else if (is_1.default.nonEmptyArray(updatedDeps)) {
        if (supportsNoInstall) {
            const updateCommands = updatedDeps
                .map((dep) => dep.depName && `devbox update ${(0, shlex_1.quote)(dep.depName)} --no-install`)
                .filter((dep) => Boolean(dep));
            if (updateCommands.length) {
                cmd.push(...updateCommands);
            }
            else {
                logger_1.logger.trace('No updated devbox packages - returning null');
                return null;
            }
        }
        else {
            cmd.push('devbox install');
        }
    }
    else {
        logger_1.logger.trace('No updated devbox packages - returning null');
        return null;
    }
    const oldLockFileContent = await (0, fs_1.readLocalFile)(lockFileName);
    if (!oldLockFileContent) {
        logger_1.logger.trace(`No ${lockFileName} found`);
        return null;
    }
    try {
        await (0, exec_1.exec)(cmd, execOptions);
        const newLockFileContent = await (0, fs_1.readLocalFile)(lockFileName);
        if (!newLockFileContent ||
            Buffer.compare(oldLockFileContent, newLockFileContent) === 0) {
            return null;
        }
        logger_1.logger.trace('Returning updated devbox.lock');
        return [
            {
                file: {
                    type: 'addition',
                    path: lockFileName,
                    contents: newLockFileContent,
                },
            },
        ];
    }
    catch (err) {
        logger_1.logger.warn({ err }, 'Error updating devbox.lock');
        return [
            {
                artifactError: {
                    lockFile: lockFileName,
                    stderr: err.message,
                },
            },
        ];
    }
}
//# sourceMappingURL=artifacts.js.map