"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handleMatching = handleMatching;
exports.createDependency = createDependency;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const jsonata_1 = tslib_1.__importDefault(require("jsonata"));
const datasource_migration_1 = require("../../../../config/migrations/custom/datasource-migration");
const logger_1 = require("../../../../logger");
const template = tslib_1.__importStar(require("../../../../util/template"));
const url_1 = require("../../../../util/url");
const utils_1 = require("../utils");
const schema_1 = require("./schema");
async function handleMatching(json, packageFile, config) {
    let results = [];
    const { matchStrings: jsonataQueries } = config;
    for (const query of jsonataQueries) {
        // won't fail as this is verified during config validation
        const jsonataExpression = (0, jsonata_1.default)(query);
        // this does not throw error, just returns undefined if no matches
        const queryResult = await jsonataExpression.evaluate(json);
        // allows empty dep object cause templates can be used to configure the required fields
        // if some issues arise then the isValidDependency call will catch them later on
        if (!queryResult || is_1.default.emptyArray(queryResult)) {
            logger_1.logger.debug({
                jsonataQuery: query,
                packageFile,
            }, 'The jsonata query returned no matches. Possible error, please check your query. Skipping');
            return [];
        }
        const parsed = schema_1.QueryResultZodSchema.safeParse(queryResult);
        if (parsed.success) {
            results = results.concat(parsed.data);
        }
        else {
            logger_1.logger.warn({ err: parsed.error, jsonataQuery: query, packageFile, queryResult }, 'Query results failed schema validation');
            return [];
        }
    }
    return results
        .map((dep) => createDependency(dep, config))
        .filter(is_1.default.truthy)
        .filter((dep) => (0, utils_1.checkIsValidDependency)(dep, packageFile, 'custom.jsonata'));
}
function createDependency(queryResult, config) {
    const dependency = {};
    for (const field of utils_1.validMatchFields) {
        const fieldTemplate = `${field}Template`;
        const tmpl = config[fieldTemplate];
        if (tmpl) {
            try {
                const compiled = template.compile(tmpl, queryResult, false);
                updateDependency(field, compiled, dependency);
            }
            catch {
                logger_1.logger.debug({ template: tmpl }, 'Error compiling template for JSONata manager');
                return null;
            }
        }
        else if (queryResult[field]) {
            updateDependency(field, queryResult[field], dependency);
        }
    }
    return dependency;
}
function updateDependency(field, value, dependency) {
    switch (field) {
        case 'registryUrl': {
            const url = (0, url_1.parseUrl)(value)?.toString();
            if (!url) {
                logger_1.logger.debug({ url: value }, 'Invalid JSONata manager registryUrl');
                break;
            }
            dependency.registryUrls = [url];
            break;
        }
        case 'datasource':
            dependency.datasource = (0, datasource_migration_1.migrateDatasource)(value);
            break;
        default:
            dependency[field] = value;
            break;
    }
    return dependency;
}
//# sourceMappingURL=utils.js.map