"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.delimiters = void 0;
exports.extractRubyVersion = extractRubyVersion;
exports.getRubyConstraint = getRubyConstraint;
exports.getBundlerConstraint = getBundlerConstraint;
exports.getLockFilePath = getLockFilePath;
const logger_1 = require("../../../logger");
const fs_1 = require("../../../util/fs");
const regex_1 = require("../../../util/regex");
exports.delimiters = ['"', "'"];
function extractRubyVersion(txt) {
    const rubyMatch = (0, regex_1.regEx)(/^ruby\s+("[^"]+"|'[^']+')\s*$/gm).exec(txt);
    if (rubyMatch?.length !== 2) {
        return null;
    }
    const quotedVersion = rubyMatch[1];
    return quotedVersion.substring(1, quotedVersion.length - 1);
}
async function getRubyConstraint(updateArtifact) {
    const { packageFileName, config, newPackageFileContent } = updateArtifact;
    const { constraints = {} } = config;
    const { ruby } = constraints;
    if (ruby) {
        logger_1.logger.debug('Using ruby constraint from config');
        return ruby;
    }
    else {
        const rubyMatch = extractRubyVersion(newPackageFileContent);
        if (rubyMatch) {
            logger_1.logger.debug('Using ruby version from gemfile');
            return rubyMatch;
        }
        for (const file of ['.ruby-version', '.tool-versions']) {
            const rubyVersion = (await (0, fs_1.readLocalFile)((0, fs_1.getSiblingFileName)(packageFileName, file), 'utf8'))?.match((0, regex_1.regEx)(/^(?:ruby(?:-|\s+))?(\d[\d.]*)/m))?.[1];
            if (rubyVersion) {
                logger_1.logger.debug(`Using ruby version specified in ${file}`);
                return rubyVersion;
            }
        }
        const lockFile = await getLockFilePath(packageFileName);
        if (lockFile) {
            const rubyVersion = (await (0, fs_1.readLocalFile)(lockFile, 'utf8'))?.match((0, regex_1.regEx)(/^ {3}ruby (\d[\d.]*)(?:[a-z]|\s|$)/m))?.[1];
            if (rubyVersion) {
                logger_1.logger.debug(`Using ruby version specified in lock file`);
                return rubyVersion;
            }
        }
    }
    return null;
}
function getBundlerConstraint(updateArtifact, existingLockFileContent) {
    const { config } = updateArtifact;
    const { constraints = {} } = config;
    const { bundler } = constraints;
    if (bundler) {
        logger_1.logger.debug('Using bundler constraint from config');
        return bundler;
    }
    else {
        const bundledWith = (0, regex_1.regEx)(/\nBUNDLED WITH\n\s+(.*?)(\n|$)/).exec(existingLockFileContent);
        if (bundledWith) {
            logger_1.logger.debug('Using bundler version specified in lockfile');
            return bundledWith[1];
        }
    }
    return null;
}
async function getLockFilePath(packageFilePath) {
    const lockFilePath = (await (0, fs_1.localPathExists)(`${packageFilePath}.lock`))
        ? `${packageFilePath}.lock`
        : `Gemfile.lock`;
    logger_1.logger.debug(`Lockfile for ${packageFilePath} found in ${lockFilePath}`);
    return lockFilePath;
}
//# sourceMappingURL=common.js.map