"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.processPackageFile = processPackageFile;
exports.extractAllPackageFiles = extractAllPackageFiles;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const logger_1 = require("../../../logger");
const fs_1 = require("../../../util/fs");
const package_file_1 = require("../npm/extract/common/package-file");
const utils_1 = require("./utils");
function matchesFileName(fileNameWithPath, fileName) {
    return (fileNameWithPath === fileName || fileNameWithPath.endsWith(`/${fileName}`));
}
async function processPackageFile(packageFile) {
    const fileContent = await (0, fs_1.readLocalFile)(packageFile, 'utf8');
    if (!fileContent) {
        logger_1.logger.warn({ fileName: packageFile }, 'Could not read file content');
        return null;
    }
    let packageJson;
    try {
        packageJson = JSON.parse(fileContent);
    }
    catch (err) {
        logger_1.logger.debug({ err }, 'Error parsing package.json');
        return null;
    }
    const result = (0, package_file_1.extractPackageJson)(packageJson, packageFile);
    if (!result) {
        logger_1.logger.debug({ packageFile }, 'No dependencies found');
        return null;
    }
    return {
        ...result,
        packageFile,
    };
}
async function extractAllPackageFiles(config, matchedFiles) {
    const packageFiles = [];
    const allLockFiles = matchedFiles.filter((file) => matchesFileName(file, 'bun.lock') || matchesFileName(file, 'bun.lockb'));
    if (allLockFiles.length === 0) {
        logger_1.logger.debug('No bun lockfiles found');
        return packageFiles;
    }
    const allPackageJson = matchedFiles.filter((file) => matchesFileName(file, 'package.json'));
    for (const lockFile of allLockFiles) {
        const packageFile = (0, fs_1.getSiblingFileName)(lockFile, 'package.json');
        const res = await processPackageFile(packageFile);
        if (res) {
            packageFiles.push({ ...res, lockFiles: [lockFile] });
        }
        // Check if package.json contains workspaces
        const workspaces = res?.managerData?.workspaces;
        if (!is_1.default.array(workspaces, is_1.default.string)) {
            continue;
        }
        logger_1.logger.debug(`Found bun workspaces in ${packageFile}`);
        const pwd = (0, fs_1.getParentDir)(packageFile);
        const workspacePackageFiles = (0, utils_1.filesMatchingWorkspaces)(pwd, allPackageJson, workspaces);
        if (workspacePackageFiles.length) {
            logger_1.logger.debug({ workspacePackageFiles }, 'Found bun workspace files');
            for (const workspaceFile of workspacePackageFiles) {
                const res = await processPackageFile(workspaceFile);
                if (res) {
                    packageFiles.push({ ...res, lockFiles: [lockFile] });
                }
            }
        }
    }
    return packageFiles;
}
//# sourceMappingURL=extract.js.map