"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.updateArtifacts = updateArtifacts;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const upath_1 = tslib_1.__importDefault(require("upath"));
const global_1 = require("../../../config/global");
const error_messages_1 = require("../../../constants/error-messages");
const logger_1 = require("../../../logger");
const exec_1 = require("../../../util/exec");
const fs_1 = require("../../../util/fs");
const rules_1 = require("../npm/post-update/rules");
const utils_1 = require("../npm/utils");
async function updateArtifacts(updateArtifact) {
    const { packageFileName, updatedDeps, newPackageFileContent, config } = updateArtifact;
    logger_1.logger.debug(`bun.updateArtifacts(${packageFileName})`);
    const { isLockFileMaintenance } = config;
    if (is_1.default.emptyArray(updatedDeps) && !isLockFileMaintenance) {
        logger_1.logger.debug('No updated bun deps - returning null');
        return null;
    }
    // Find the first bun dependency in order to handle mixed manager updates
    const lockFileName = updatedDeps.find((dep) => dep.manager === 'bun')
        ?.lockFiles?.[0];
    if (!lockFileName) {
        logger_1.logger.debug(`No ${lockFileName} found`);
        return null;
    }
    const oldLockFileContent = await (0, fs_1.readLocalFile)(lockFileName);
    if (!oldLockFileContent) {
        logger_1.logger.debug(`No ${lockFileName} found`);
        return null;
    }
    const pkgFileDir = upath_1.default.dirname(packageFileName);
    const npmrcContent = await (0, utils_1.getNpmrcContent)(pkgFileDir);
    const { additionalNpmrcContent } = (0, rules_1.processHostRules)();
    await (0, utils_1.updateNpmrcContent)(pkgFileDir, npmrcContent, additionalNpmrcContent);
    try {
        await (0, fs_1.writeLocalFile)(packageFileName, newPackageFileContent);
        if (isLockFileMaintenance) {
            await (0, fs_1.deleteLocalFile)(lockFileName);
        }
        let cmd = 'bun install';
        if (!global_1.GlobalConfig.get('allowScripts') || config.ignoreScripts) {
            cmd += ' --ignore-scripts';
        }
        const execOptions = {
            cwdFile: packageFileName,
            docker: {},
            toolConstraints: [
                {
                    toolName: 'bun',
                    constraint: updateArtifact?.config?.constraints?.bun,
                },
            ],
        };
        await (0, exec_1.exec)(cmd, execOptions);
        await (0, utils_1.resetNpmrcContent)(pkgFileDir, npmrcContent);
        const newLockFileContent = await (0, fs_1.readLocalFile)(lockFileName);
        if (!newLockFileContent ||
            Buffer.compare(oldLockFileContent, newLockFileContent) === 0) {
            return null;
        }
        return [
            {
                file: {
                    type: 'addition',
                    path: lockFileName,
                    contents: newLockFileContent,
                },
            },
        ];
    }
    catch (err) {
        if (err.message === error_messages_1.TEMPORARY_ERROR) {
            throw err;
        }
        logger_1.logger.warn({ lockfile: lockFileName, err }, `Failed to update lock file`);
        return [
            {
                artifactError: {
                    lockFile: lockFileName,
                    stderr: err.message,
                },
            },
        ];
    }
}
//# sourceMappingURL=artifacts.js.map