"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BUILDPACK_REGISTRY_PREFIX = exports.DOCKER_PREFIX = void 0;
exports.isDockerRef = isDockerRef;
exports.isBuildpackRegistryRef = isBuildpackRegistryRef;
exports.extractPackageFile = extractPackageFile;
exports.getDep = getDep;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const logger_1 = require("../../../logger");
const regex_1 = require("../../../util/regex");
const buildpacks_registry_1 = require("../../datasource/buildpacks-registry");
const semver_1 = require("../../versioning/semver");
const extract_1 = require("../dockerfile/extract");
const schema_1 = require("./schema");
exports.DOCKER_PREFIX = (0, regex_1.regEx)(/^docker:\/?\//);
const dockerRef = (0, regex_1.regEx)(/^((?:[a-z\d](?:[a-z\d-]{0,61}[a-z\d])?(?:\.[a-z\d](?:[a-z\d-]{0,61}[a-z\d])?)*)(?::\d{2,5}\/)?)?[a-z\d]+((\.|_|__|-+)[a-z\d]+)*(\/[a-z\d]+((\.|_|__|-+)[a-z\d]+)*)*(?::(\w[\w.-]{0,127})(?:@sha256:[A-Fa-f\d]{32,})?|@sha256:[A-Fa-f\d]{32,})$/);
function isDockerRef(ref) {
    if (ref.startsWith('docker:/') || dockerRef.test(ref)) {
        return true;
    }
    return false;
}
exports.BUILDPACK_REGISTRY_PREFIX = 'urn:cnb:registry:';
const buildpackRegistryId = (0, regex_1.regEx)(/^[a-z0-9\-.]+\/[a-z0-9\-.]+(?:@(?<version>.+))?$/);
function isBuildpackRegistryId(ref) {
    const bpRegistryMatch = buildpackRegistryId.exec(ref);
    if (!bpRegistryMatch) {
        return false;
    }
    else if (!bpRegistryMatch.groups?.version) {
        return true;
    }
    return (0, semver_1.isVersion)(bpRegistryMatch.groups.version);
}
function isBuildpackRegistryRef(ref) {
    return (isBuildpackRegistryId(ref) || ref.startsWith(exports.BUILDPACK_REGISTRY_PREFIX));
}
function parseProjectToml(content, packageFile) {
    const res = schema_1.ProjectDescriptorToml.safeParse(content);
    if (res.success) {
        return res.data;
    }
    logger_1.logger.debug({ packageFile, err: res.error }, 'Failed to parse buildpacks project descriptor TOML');
    return null;
}
function extractPackageFile(content, packageFile, config) {
    const deps = [];
    const descriptor = parseProjectToml(content, packageFile);
    if (!descriptor) {
        return null;
    }
    if (descriptor.io?.buildpacks?.builder &&
        isDockerRef(descriptor.io.buildpacks.builder)) {
        const dep = (0, extract_1.getDep)(descriptor.io.buildpacks.builder.replace(exports.DOCKER_PREFIX, ''), true, config.registryAliases);
        logger_1.logger.trace({
            depName: dep.depName,
            currentValue: dep.currentValue,
            currentDigest: dep.currentDigest,
        }, 'Cloud Native Buildpacks builder');
        deps.push({ ...dep, commitMessageTopic: 'builder {{depName}}' });
    }
    if (descriptor.io?.buildpacks?.group &&
        is_1.default.array(descriptor.io.buildpacks.group)) {
        for (const group of descriptor.io.buildpacks.group) {
            if ((0, schema_1.isBuildpackByURI)(group) && isDockerRef(group.uri)) {
                const dep = (0, extract_1.getDep)(group.uri.replace(exports.DOCKER_PREFIX, ''), true, config.registryAliases);
                logger_1.logger.trace({
                    depName: dep.depName,
                    currentValue: dep.currentValue,
                    currentDigest: dep.currentDigest,
                }, 'Cloud Native Buildpack');
                deps.push(dep);
            }
            else if ((0, schema_1.isBuildpackByURI)(group) && isBuildpackRegistryRef(group.uri)) {
                const dep = getDep(group.uri.replace(exports.BUILDPACK_REGISTRY_PREFIX, ''));
                if (dep) {
                    deps.push(dep);
                }
            }
            else if ((0, schema_1.isBuildpackByName)(group)) {
                const version = group.version;
                if (version) {
                    const dep = {
                        datasource: buildpacks_registry_1.BuildpacksRegistryDatasource.id,
                        currentValue: version,
                        packageName: group.id,
                    };
                    deps.push(dep);
                }
            }
        }
    }
    if (!deps.length) {
        return null;
    }
    return { deps };
}
function getDep(currentFrom) {
    if (currentFrom.includes('@')) {
        const dep = {
            datasource: buildpacks_registry_1.BuildpacksRegistryDatasource.id,
            packageName: currentFrom.split('@')[0],
            autoReplaceStringTemplate: '{{depName}}{{#if newValue}}:{{newValue}}{{/if}}{{#if newDigest}}@{{newDigest}}{{/if}}',
        };
        const version = currentFrom.split('@')[1];
        dep.currentValue = version;
        return dep;
    }
    return null;
}
//# sourceMappingURL=extract.js.map